<?php
/**
 * Vérification de majorité par carte d'identité française
 * Version 4.0 - Support ancien ET nouveau format CNI (2021+)
 * 
 * Installation requise:
 * sudo apt install tesseract-ocr tesseract-ocr-fra imagemagick
 * sudo mkdir -p /tmp/id_verify && sudo chmod 777 /tmp/id_verify
 * 
 * Nouveau format (depuis août 2021):
 * - Format carte bancaire (ID-1): 85.6 x 53.98 mm
 * - MRZ TD1 sur 3 lignes de 30 caractères
 * - Ligne 2: Numéro doc + Date naissance (AAMMJJ) + Sexe + Date expiration
 * 
 * Ancien format (avant 2021):
 * - Format ID-2: 105 x 74 mm  
 * - MRZ sur 2 lignes de 36 caractères
 * - Date format: Né(e) le JJ.MM.AAAA
 */

class IDCardVerifier
{
    private string $uploadDir = '/tmp/id_verify/';
    private int $minAge = 18;
    private array $debug = [];
    private ?string $detectedFormat = null;

    public function __construct()
    {
        if (!file_exists($this->uploadDir)) {
            mkdir($this->uploadDir, 0755, true);
        }
    }

    /**
     * Prétraitement standard de l'image
     */
    private function preprocessImage(string $inputPath, string $outputPath, int $rotation = 0): bool
    {
        $rotateCmd = $rotation !== 0 ? "-rotate $rotation" : "";
        $cmd = "convert '$inputPath' $rotateCmd -colorspace Gray -normalize -sharpen 0x1 -density 300 '$outputPath' 2>&1";
        exec($cmd, $output, $returnCode);
        return $returnCode === 0 && file_exists($outputPath);
    }

    /**
     * Prétraitement optimisé pour la zone MRZ
     */
    private function preprocessForMRZ(string $inputPath, string $outputPath, int $rotation = 0): bool
    {
        $rotateCmd = $rotation !== 0 ? "-rotate $rotation" : "";
        // Contraste élevé + binarisation pour MRZ
        $cmd = "convert '$inputPath' $rotateCmd -colorspace Gray -contrast-stretch 5%x5% -threshold 50% -density 300 '$outputPath' 2>&1";
        exec($cmd, $output, $returnCode);
        return $returnCode === 0 && file_exists($outputPath);
    }

    /**
     * Exécute Tesseract OCR avec plusieurs configurations
     */
    private function runOCR(string $imagePath, bool $mrzMode = false): string
    {
        $outputBase = $this->uploadDir . uniqid('ocr_');
        $text = '';
        
        // PSM modes: 6=bloc uniforme, 3=auto, 4=colonne, 11=texte sparse, 13=raw line
        $psmModes = $mrzMode ? [6, 13, 3] : [6, 3, 4, 11];
        $lang = $mrzMode ? 'eng' : 'fra+eng'; // MRZ est en caractères latins standards
        
        foreach ($psmModes as $psm) {
            $cmd = "tesseract '$imagePath' '$outputBase' -l $lang --psm $psm 2>&1";
            exec($cmd, $output, $returnCode);
            $textFile = $outputBase . '.txt';
            if (file_exists($textFile)) {
                $content = file_get_contents($textFile);
                $text .= "\n" . $content;
                unlink($textFile);
                if ($this->hasDatePattern($content)) break;
            }
        }
        return $text;
    }

    /**
     * Vérifie si le texte contient des patterns de date
     */
    private function hasDatePattern(string $text): bool
    {
        $patterns = [
            '/\d{2}[\s\.\/]\d{2}[\s\.\/]\d{4}/',  // JJ.MM.AAAA ou JJ MM AAAA
            '/\d{2}[\s\.\/]\d{2}[\s\.\/]\d{2}/',  // JJ.MM.AA
            '/n[ée]e?\s/i',
            '/naissance/i',
            '/IDFRA/i',                            // Nouveau format MRZ
            '/ID[A-Z]{3}/i',                       // Code pays MRZ
            '/\d{6,7}[FM<]/i',                     // Date MRZ + sexe
        ];
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $text)) return true;
        }
        return false;
    }

    /**
     * Score de qualité OCR - adapté aux deux formats
     */
    private function getOCRQualityScore(string $text): int
    {
        $score = 0;
        $textLower = strtolower($text);
        $textUpper = strtoupper($text);
        
        // Mots clés communs aux deux formats
        $commonKeywords = [
            'république' => 10, 'republique' => 10, 'francaise' => 10, 'française' => 10,
            'carte' => 8, 'identité' => 10, 'identite' => 10,
            'nationale' => 8, 'nom' => 5, 'prénom' => 5, 'prenom' => 5,
            'née' => 8, 'nee' => 8, 'né' => 8, 'ne le' => 8,
            'naissance' => 10, 'sexe' => 5, 'nationalité' => 8,
            'nationalite' => 8, 'signature' => 5,
        ];
        
        // Mots clés spécifiques au NOUVEAU format (2021+)
        $newFormatKeywords = [
            'date de naissance' => 12,
            'lieu de naissance' => 10,
            'date d\'expiration' => 10,
            'taille' => 5,
            'card access number' => 15,
            'can' => 5,
        ];
        
        // Patterns MRZ nouveau format (3 lignes de 30 caractères)
        $newMRZPatterns = [
            '/IDFRA[A-Z0-9<]{24}/i' => 20,      // Ligne 1 nouveau format
            '/[A-Z0-9<]{10}FRA[0-9]{6}[FM<]/i' => 25, // Ligne 2 avec date
        ];
        
        // Patterns MRZ ancien format (2 lignes de 36 caractères)
        $oldMRZPatterns = [
            '/IDFRA[A-Z0-9<]{30}/i' => 20,      // Ligne 1 ancien format
        ];
        
        // Mots clés spécifiques à l'ANCIEN format
        $oldFormatKeywords = [
            'domicile' => 8,
            'délivrée' => 8, 'delivree' => 8,
            'préfecture' => 8, 'prefecture' => 8,
            'valable jusqu' => 10,
        ];
        
        foreach ($commonKeywords as $word => $points) {
            if (strpos($textLower, $word) !== false) $score += $points;
        }
        
        foreach ($newFormatKeywords as $word => $points) {
            if (strpos($textLower, $word) !== false) {
                $score += $points;
                $this->detectedFormat = 'new';
            }
        }
        
        foreach ($oldFormatKeywords as $word => $points) {
            if (strpos($textLower, $word) !== false) {
                $score += $points;
                if ($this->detectedFormat !== 'new') $this->detectedFormat = 'old';
            }
        }
        
        foreach ($newMRZPatterns as $pattern => $points) {
            if (preg_match($pattern, $textUpper)) {
                $score += $points;
                $this->detectedFormat = 'new';
            }
        }
        
        foreach ($oldMRZPatterns as $pattern => $points) {
            if (preg_match($pattern, $textUpper)) {
                $score += $points;
                if ($this->detectedFormat !== 'new') $this->detectedFormat = 'old';
            }
        }

        // Bonus pour patterns de date
        if (preg_match('/\d{2}[\s\.]\d{2}[\s\.]\d{4}/', $text)) $score += 20;
        
        // Pénalité pour caractères bizarres
        $weirdChars = preg_match_all('/[|{}[\]@#$%&*§µ£]/', $text);
        return max(0, $score - $weirdChars * 2);
    }

    /**
     * Trouve la meilleure rotation pour l'image
     */
    private function findBestRotation(string $inputPath): array
    {
        $rotations = [0, 90, 180, 270];
        $bestRotation = 0;
        $bestScore = 0;
        $allResults = [];

        foreach ($rotations as $rotation) {
            $tempFile = $this->uploadDir . uniqid("rot{$rotation}_") . '.png';
            $mrzFile = $this->uploadDir . uniqid("mrz{$rotation}_") . '.png';
            
            if ($this->preprocessImage($inputPath, $tempFile, $rotation)) {
                // OCR standard
                $text = $this->runOCR($tempFile);
                
                // OCR optimisé MRZ
                if ($this->preprocessForMRZ($inputPath, $mrzFile, $rotation)) {
                    $text .= "\n" . $this->runOCR($mrzFile, true);
                    @unlink($mrzFile);
                }
                
                $score = $this->getOCRQualityScore($text);
                
                $allResults[$rotation] = [
                    'score' => $score,
                    'text' => $text,
                    'preview' => substr(preg_replace('/\s+/', ' ', $text), 0, 150),
                    'format' => $this->detectedFormat
                ];

                if ($score > $bestScore) {
                    $bestScore = $score;
                    $bestRotation = $rotation;
                }

                @unlink($tempFile);
            }
        }

        $this->debug['rotation_scores'] = array_map(fn($r) => [
            'score' => $r['score'], 
            'preview' => $r['preview'],
            'format' => $r['format']
        ], $allResults);
        $this->debug['best_rotation'] = $bestRotation;
        $this->debug['best_score'] = $bestScore;
        $this->debug['detected_format'] = $allResults[$bestRotation]['format'] ?? 'unknown';

        return [
            'rotation' => $bestRotation,
            'score' => $bestScore,
            'text' => $allResults[$bestRotation]['text'] ?? '',
            'format' => $allResults[$bestRotation]['format'] ?? 'unknown'
        ];
    }

    /**
     * Extrait la date de naissance - NOUVEAU FORMAT MRZ (TD1 - 3 lignes)
     * Ligne 2 format: [Numéro 9 chars][Check 1][AAMMJJ naissance][Check 1][Sexe F/M/<][AAMMJJ expiration][Check 1][Nationalité 3]...
     */
    private function extractFromNewMRZ(string $text): ?string
    {
        $text = strtoupper(preg_replace('/\s+/', '', $text));
        
        // Pattern ligne 2 nouveau format CNI française
        // Format: XXXXXXXXXC AAMMJJ C S AAMMJJ C FRA...
        $patterns = [
            // Pattern complet ligne 2
            '/([A-Z0-9<]{9})(\d)(\d{2})(0[1-9]|1[0-2])(0[1-9]|[12]\d|3[01])(\d)([FM<])/',
            // Pattern simplifié avec date
            '/(\d{2})(0[1-9]|1[0-2])(0[1-9]|[12]\d|3[01])\d[FM<]/',
            // IDFRA suivi de données
            '/IDFRA.{20,}(\d{2})(0[1-9]|1[0-2])(0[1-9]|[12]\d|3[01])/',
        ];
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $text, $matches)) {
                $this->debug['mrz_pattern'] = $pattern;
                $this->debug['mrz_matches'] = $matches;
                
                // Extraction selon le pattern
                if (count($matches) >= 6 && strlen($matches[3]) == 2) {
                    // Pattern complet
                    $year = (int)$matches[3];
                    $month = $matches[4];
                    $day = $matches[5];
                } elseif (count($matches) >= 4) {
                    // Patterns simplifiés
                    $year = (int)$matches[1];
                    $month = $matches[2];
                    $day = $matches[3];
                } else {
                    continue;
                }
                
                // Conversion année 2 chiffres -> 4 chiffres
                $year = $year > 30 ? 1900 + $year : 2000 + $year;
                
                if ($this->isValidDate($day, $month, $year)) {
                    $this->detectedFormat = 'new';
                    return "$year-$month-$day";
                }
            }
        }
        
        return null;
    }

    /**
     * Extrait la date de naissance - ANCIEN FORMAT MRZ (ID-2 - 2 lignes)
     * Ligne 2 format similaire mais 36 caractères
     */
    private function extractFromOldMRZ(string $text): ?string
    {
        $text = strtoupper(preg_replace('/\s+/', '', $text));
        
        // Pattern ancien format (ligne 2, 36 chars)
        $patterns = [
            // Format standard ancien MRZ
            '/(\d{2})(0[1-9]|1[0-2])(0[1-9]|[12]\d|3[01])\d{1,2}[FM]/',
            // Avec séparateurs < 
            '/[0-9<]{6,}(\d{2})(0[1-9]|1[0-2])(0[1-9]|[12]\d|3[01])/',
        ];
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $text, $matches)) {
                $this->debug['old_mrz_pattern'] = $pattern;
                $this->debug['old_mrz_matches'] = $matches;
                
                $year = (int)$matches[1];
                $month = $matches[2];
                $day = $matches[3];
                
                $year = $year > 30 ? 1900 + $year : 2000 + $year;
                
                if ($this->isValidDate($day, $month, $year)) {
                    $this->detectedFormat = 'old';
                    return "$year-$month-$day";
                }
            }
        }
        
        return null;
    }

    /**
     * Extrait la date de naissance du texte OCR (hors MRZ)
     */
    private function extractFromVisualText(string $text): ?string
    {
        $originalText = $text;
        $text = strtolower($text);
        $text = preg_replace('/\s+/', ' ', $text);
        
        $this->debug['cleaned_text'] = substr($text, 0, 500);

        // Mois en lettres
        $monthNames = [
            'janvier' => '01', 'janv' => '01', 'jan' => '01',
            'février' => '02', 'fevrier' => '02', 'fév' => '02', 'fev' => '02',
            'mars' => '03', 'mar' => '03',
            'avril' => '04', 'avr' => '04',
            'mai' => '05',
            'juin' => '06', 'jun' => '06',
            'juillet' => '07', 'juil' => '07', 'jul' => '07',
            'août' => '08', 'aout' => '08', 'aou' => '08',
            'septembre' => '09', 'sept' => '09', 'sep' => '09',
            'octobre' => '10', 'oct' => '10',
            'novembre' => '11', 'nov' => '11',
            'décembre' => '12', 'decembre' => '12', 'déc' => '12', 'dec' => '12',
        ];

        // === NOUVEAU FORMAT CNI 2021+ ===
        // Chercher spécifiquement le pattern "DATE DE NAISS" suivi d'une date
        // Le texte OCR peut donner: "date de naiss / date of birth" puis "01 041995" ou "01 04 1995"
        
        // Pattern 1: DATE DE NAISS avec date qui suit (même séparée par du texte)
        if (preg_match('/date\s*de\s*naiss.*?(\d{1,2})[\s\.]*(\d{2})[\s\.]*(\d{4})/is', $text, $matches)) {
            $this->debug['visual_pattern_matched'] = 'new_cni_date_naiss_flexible';
            $this->debug['visual_matches'] = $matches;
            $day = str_pad($matches[1], 2, '0', STR_PAD_LEFT);
            $month = str_pad($matches[2], 2, '0', STR_PAD_LEFT);
            $year = $matches[3];
            if ($this->isValidDate($day, $month, $year)) {
                return "$year-$month-$day";
            }
        }
        
        // Pattern 2: Chercher "fra" (nationalité) suivi d'une date (typique du nouveau format)
        if (preg_match('/\bfra\b.*?(\d{1,2})[\s\.]*(\d{2})[\s\.]*(\d{4})/is', $text, $matches)) {
            $day = str_pad($matches[1], 2, '0', STR_PAD_LEFT);
            $month = str_pad($matches[2], 2, '0', STR_PAD_LEFT);
            $year = $matches[3];
            // Vérifier que c'est une date de naissance plausible (pas une date d'expiration)
            $yearInt = (int)$year;
            if ($yearInt >= 1920 && $yearInt <= 2020 && $this->isValidDate($day, $month, $year)) {
                $this->debug['visual_pattern_matched'] = 'fra_followed_by_date';
                $this->debug['visual_matches'] = $matches;
                return "$year-$month-$day";
            }
        }

        // Patterns par ordre de priorité
        $patterns = [
            // NOUVEAU FORMAT CNI 2021+: "DATE DE NAISS" suivi de JJ MM AAAA ou JJ MMAAAA
            '/date\s*de\s*naiss[^\d]*(\d{1,2})[\s\.\/-]*(\d{2})[\s\.\/-]*(\d{4})/i' => 'new_cni_date_naiss',
            
            // NOUVEAU FORMAT: "Date de naissance JJ.MM.AAAA" ou "JJ MM AAAA"
            '/date\s*(?:de)?\s*naissance\s*[:\s]*(\d{1,2})[\s\.\/-]+(\d{1,2})[\s\.\/-]+(\d{4})/i' => 'new_date_naissance',
            
            // Format avec espaces variables et collés (01 041995)
            '/(\d{2})[\s\.]*(\d{2})[\s\.]*(\d{4})/' => 'date_collee',
            
            // ANCIEN FORMAT: "Né(e) le JJ.MM.AAAA" ou "Né(e) le JJ MOIS AAAA"
            '/n[ée]e?\s*(?:le)?\s*[:\s]*(\d{1,2})[\s\.\/-]+(\d{1,2})[\s\.\/-]+(\d{4})/i' => 'nee_le_num',
            
            // Avec mois en lettres
            '/n[ée]e?\s*(?:le)?\s*(\d{1,2})\s*(janvier|février|fevrier|mars|avril|mai|juin|juillet|août|aout|septembre|octobre|novembre|décembre|decembre)\s*(\d{4})/i' => 'nee_le_mois',
            
            // Date de naissance avec mois en lettres
            '/date\s*(?:de)?\s*naissance\s*[:\s]*(\d{1,2})\s*(janvier|février|fevrier|mars|avril|mai|juin|juillet|août|aout|septembre|octobre|novembre|décembre|decembre)\s*(\d{4})/i' => 'date_naissance_mois',
            
            // Naissance seul
            '/naissance\s*[:\s]+(\d{1,2})[\s\.\/-]+(\d{1,2})[\s\.\/-]+(\d{4})/i' => 'naissance',
            
            // Format JJ MMAAAA (mois et année collés)
            '/(\d{1,2})\s+(\d{2})(\d{4})/' => 'jour_moisannee_colles',

            // Format générique JJ.MM.AAAA ou JJ/MM/AAAA ou JJ-MM-AAAA
            '/(\d{1,2})[\.\/-](\d{1,2})[\.\/-]((?:19|20)\d{2})/' => 'generique_separateur',
            
            // Format générique avec espaces
            '/(\d{1,2})\s+(\d{1,2})\s+((?:19|20)\d{2})/' => 'generique_espace',
        ];

        foreach ($patterns as $pattern => $patternName) {
            if (preg_match($pattern, $text, $matches)) {
                $this->debug['visual_pattern_matched'] = $patternName;
                $this->debug['visual_matches'] = $matches;

                if (strpos($patternName, 'mois') !== false && strpos($patternName, 'colles') === false) {
                    // Pattern avec mois en lettres
                    $day = str_pad($matches[1], 2, '0', STR_PAD_LEFT);
                    $monthLower = strtolower($matches[2]);
                    $month = null;
                    foreach ($monthNames as $name => $num) {
                        if (strpos($monthLower, $name) === 0) {
                            $month = $num;
                            break;
                        }
                    }
                    if (!$month) continue;
                    $year = $matches[3];
                } elseif ($patternName === 'jour_moisannee_colles') {
                    // Pattern JJ MMAAAA (mois et année collés)
                    $day = str_pad($matches[1], 2, '0', STR_PAD_LEFT);
                    $month = $matches[2];
                    $year = $matches[3];
                } else {
                    // Pattern numérique standard
                    $day = str_pad($matches[1], 2, '0', STR_PAD_LEFT);
                    $month = str_pad($matches[2], 2, '0', STR_PAD_LEFT);
                    $year = $matches[3];
                }

                if ($this->isValidDate($day, $month, $year)) {
                    return "$year-$month-$day";
                }
            }
        }

        return null;
    }

    /**
     * Valide une date
     */
    private function isValidDate(string $day, string $month, int $year): bool
    {
        $monthInt = (int)$month;
        $dayInt = (int)$day;
        
        if ($monthInt < 1 || $monthInt > 12) return false;
        if ($dayInt < 1 || $dayInt > 31) return false;
        if ($year < 1900 || $year > (int)date('Y')) return false;
        
        return checkdate($monthInt, $dayInt, $year);
    }

    /**
     * Extraction principale de la date de naissance
     */
    private function extractBirthDate(string $text, string $detectedFormat): ?string
    {
        // 1. Essayer d'abord le MRZ nouveau format
        $date = $this->extractFromNewMRZ($text);
        if ($date) {
            $this->debug['extraction_method'] = 'new_mrz';
            return $date;
        }
        
        // 2. Essayer le MRZ ancien format
        $date = $this->extractFromOldMRZ($text);
        if ($date) {
            $this->debug['extraction_method'] = 'old_mrz';
            return $date;
        }
        
        // 3. Essayer l'extraction visuelle (texte lisible)
        $date = $this->extractFromVisualText($text);
        if ($date) {
            $this->debug['extraction_method'] = 'visual';
            return $date;
        }
        
        return null;
    }

    /**
     * Calcule l'âge
     */
    private function calculateAge(string $birthDate): int
    {
        return (new DateTime())->diff(new DateTime($birthDate))->y;
    }

    /**
     * Extrait le nom et prénom du texte OCR
     */
    private function extractNameInfo(string $text): array
    {
        $result = [
            'nom' => null,
            'prenom' => null,
            'sexe' => null,
            'lieu_naissance' => null,
        ];
        
        $text = preg_replace('/\s+/', ' ', $text);
        
        // Extraction du NOM (après "NOM" ou "Surname")
        if (preg_match('/(?:nom|surname)\s*[:\s\/]*([A-ZÉÈÊËÀÂÄÙÛÜÔÖÎÏÇ][A-ZÉÈÊËÀÂÄÙÛÜÔÖÎÏÇa-zéèêëàâäùûüôöîïç\-\s]{1,30}?)(?=\s*(?:prénom|prenom|given|né|nee|sexe|$))/i', $text, $matches)) {
            $nom = trim($matches[1]);
            $nom = preg_replace('/[^A-ZÉÈÊËÀÂÄÙÛÜÔÖÎÏÇa-zéèêëàâäùûüôöîïç\-\s]/', '', $nom);
            if (strlen($nom) >= 2) {
                $result['nom'] = mb_strtoupper(trim($nom));
            }
        }
        
        // Extraction du PRÉNOM (après "Prénom" ou "Given names")
        if (preg_match('/(?:prénom|prenom|given\s*names?)\s*[:\s\/]*([A-ZÉÈÊËÀÂÄÙÛÜÔÖÎÏÇa-zéèêëàâäùûüôöîïç][A-ZÉÈÊËÀÂÄÙÛÜÔÖÎÏÇa-zéèêëàâäùûüôöîïç\-\s,]{1,50}?)(?=\s*(?:sexe|sex|né|nee|national|date|lieu|$))/i', $text, $matches)) {
            $prenom = trim($matches[1]);
            $prenom = preg_replace('/[^A-ZÉÈÊËÀÂÄÙÛÜÔÖÎÏÇa-zéèêëàâäùûüôöîïç\-\s,]/', '', $prenom);
            // Prendre le premier prénom si plusieurs
            $prenoms = preg_split('/[,\s]+/', $prenom);
            if (!empty($prenoms[0]) && strlen($prenoms[0]) >= 2) {
                $result['prenom'] = ucfirst(mb_strtolower(trim($prenoms[0])));
            }
        }
        
        // Extraction du SEXE
        if (preg_match('/(?:sexe|sex)\s*[:\s\/]*([FM])\b/i', $text, $matches)) {
            $result['sexe'] = strtoupper($matches[1]);
        }
        
        // Extraction du LIEU DE NAISSANCE
        if (preg_match('/(?:lieu\s*(?:de)?\s*naissance|place\s*of\s*birth)\s*[:\s\/]*([A-ZÉÈÊËÀÂÄÙÛÜÔÖÎÏÇa-zéèêëàâäùûüôöîïç][A-ZÉÈÊËÀÂÄÙÛÜÔÖÎÏÇa-zéèêëàâäùûüôöîïç\-\s]{1,30}?)(?=\s*(?:nom|date|sexe|national|$))/i', $text, $matches)) {
            $lieu = trim($matches[1]);
            $lieu = preg_replace('/[^A-ZÉÈÊËÀÂÄÙÛÜÔÖÎÏÇa-zéèêëàâäùûüôöîïç\-\s]/', '', $lieu);
            if (strlen($lieu) >= 2) {
                $result['lieu_naissance'] = mb_strtoupper(trim($lieu));
            }
        }
        
        $this->debug['extracted_name_info'] = $result;
        
        return $result;
    }

    /**
     * Vérifie le document
     */
    public function verify(array $uploadedFile): array
    {
        $result = [
            'success' => false,
            'is_adult' => null,
            'age' => null,
            'birth_date' => null,
            'format' => 'unknown',
            'message' => '',
            'debug' => []
        ];

        // Validation du fichier
        if (!isset($uploadedFile['tmp_name']) || !is_uploaded_file($uploadedFile['tmp_name'])) {
            $result['message'] = 'Aucun fichier uploadé';
            return $result;
        }

        $allowedTypes = ['image/jpeg', 'image/png', 'image/webp', 'image/bmp', 'image/gif'];
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = finfo_file($finfo, $uploadedFile['tmp_name']);
        finfo_close($finfo);

        if (!in_array($mimeType, $allowedTypes)) {
            $result['message'] = 'Format non supporté. Utilisez JPG, PNG, WebP ou BMP.';
            return $result;
        }

        // Copie du fichier
        $extension = pathinfo($uploadedFile['name'], PATHINFO_EXTENSION) ?: 'jpg';
        $tempFile = $this->uploadDir . uniqid('id_') . '.' . $extension;
        move_uploaded_file($uploadedFile['tmp_name'], $tempFile);

        // Trouver la meilleure rotation
        $bestResult = $this->findBestRotation($tempFile);
        $this->debug['ocr_raw'] = substr($bestResult['text'], 0, 1000);

        // Si score trop bas, on essaie avec une amélioration supplémentaire
        if ($bestResult['score'] < 20) {
            $this->debug['low_score_retry'] = true;
            $enhancedFile = $this->uploadDir . uniqid('enhanced_') . '.png';
            exec("convert '$tempFile' -rotate {$bestResult['rotation']} -colorspace Gray -contrast-stretch 2%x2% -sharpen 0x2 -density 300 '$enhancedFile' 2>&1");
            if (file_exists($enhancedFile)) {
                $bestResult['text'] .= "\n" . $this->runOCR($enhancedFile);
                $bestResult['text'] .= "\n" . $this->runOCR($enhancedFile, true); // MRZ mode
                @unlink($enhancedFile);
            }
        }

        // Extraction de la date
        $birthDate = $this->extractBirthDate($bestResult['text'], $bestResult['format']);
        
        // Extraction du nom et prénom
        $nameInfo = $this->extractNameInfo($bestResult['text']);

        // Nettoyage
        @unlink($tempFile);

        if (!$birthDate) {
            $result['message'] = 'Impossible d\'extraire la date de naissance. Assurez-vous que la carte est bien lisible et que le recto (avec la date de naissance) est visible.';
            $result['debug'] = $this->debug;
            return $result;
        }

        // Calcul de l'âge
        $age = $this->calculateAge($birthDate);
        $result['success'] = true;
        $result['is_adult'] = $age >= $this->minAge;
        $result['age'] = $age;
        $result['birth_date'] = $birthDate;
        $result['nom'] = $nameInfo['nom'];
        $result['prenom'] = $nameInfo['prenom'];
        $result['sexe'] = $nameInfo['sexe'];
        $result['lieu_naissance'] = $nameInfo['lieu_naissance'];
        $result['format'] = $this->detectedFormat ?? $bestResult['format'] ?? 'unknown';
        $result['message'] = $result['is_adult'] 
            ? "✓ Personne majeure ($age ans)" 
            : "✗ Personne mineure ($age ans)";
        $result['debug'] = $this->debug;

        return $result;
    }
}

// Traitement de la requête
$result = null;
$showDebug = isset($_GET['debug']);

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['id_card'])) {
    $verifier = new IDCardVerifier();
    $result = $verifier->verify($_FILES['id_card']);
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Vérification d'identité sécurisée</title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&family=JetBrains+Mono:wght@400;500&display=swap" rel="stylesheet">
    <style>
        :root {
            --primary: #6366f1;
            --primary-glow: rgba(99, 102, 241, 0.4);
            --success: #10b981;
            --success-glow: rgba(16, 185, 129, 0.4);
            --danger: #ef4444;
            --danger-glow: rgba(239, 68, 68, 0.4);
            --warning: #f59e0b;
            --bg-dark: #0a0a0f;
            --bg-card: rgba(15, 15, 25, 0.8);
            --border: rgba(255, 255, 255, 0.08);
            --text: #e2e8f0;
            --text-muted: #64748b;
        }

        * { box-sizing: border-box; margin: 0; padding: 0; }

        body {
            font-family: 'Inter', sans-serif;
            background: var(--bg-dark);
            min-height: 100vh;
            color: var(--text);
            overflow-x: hidden;
        }

        /* Animated background */
        .bg-animation {
            position: fixed;
            top: 0; left: 0;
            width: 100%; height: 100%;
            z-index: -1;
            overflow: hidden;
        }

        .bg-animation::before {
            content: '';
            position: absolute;
            top: -50%; left: -50%;
            width: 200%; height: 200%;
            background: 
                radial-gradient(circle at 20% 80%, rgba(99, 102, 241, 0.15) 0%, transparent 50%),
                radial-gradient(circle at 80% 20%, rgba(16, 185, 129, 0.1) 0%, transparent 50%),
                radial-gradient(circle at 40% 40%, rgba(139, 92, 246, 0.1) 0%, transparent 40%);
            animation: bgRotate 30s linear infinite;
        }

        @keyframes bgRotate {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        .grid-overlay {
            position: fixed;
            top: 0; left: 0;
            width: 100%; height: 100%;
            background-image: 
                linear-gradient(rgba(99, 102, 241, 0.03) 1px, transparent 1px),
                linear-gradient(90deg, rgba(99, 102, 241, 0.03) 1px, transparent 1px);
            background-size: 50px 50px;
            z-index: -1;
        }

        .container {
            max-width: 480px;
            margin: 0 auto;
            padding: 40px 20px;
            position: relative;
        }

        /* Header */
        .header {
            text-align: center;
            margin-bottom: 40px;
        }

        .logo {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            width: 80px; height: 80px;
            background: linear-gradient(135deg, var(--primary), #8b5cf6);
            border-radius: 24px;
            margin-bottom: 24px;
            position: relative;
            box-shadow: 0 20px 40px -10px var(--primary-glow);
        }

        .logo::before {
            content: '';
            position: absolute;
            inset: -2px;
            background: linear-gradient(135deg, var(--primary), #8b5cf6, var(--success));
            border-radius: 26px;
            z-index: -1;
            opacity: 0.5;
            filter: blur(10px);
        }

        .logo svg {
            width: 40px; height: 40px;
            color: white;
        }

        .header h1 {
            font-size: 1.75rem;
            font-weight: 600;
            margin-bottom: 8px;
            background: linear-gradient(135deg, #fff, #94a3b8);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        .header p {
            color: var(--text-muted);
            font-size: 0.95rem;
        }

        /* Security badges */
        .security-badges {
            display: flex;
            justify-content: center;
            gap: 12px;
            margin-top: 16px;
            flex-wrap: wrap;
        }

        .badge {
            display: flex;
            align-items: center;
            gap: 6px;
            padding: 6px 12px;
            background: rgba(255, 255, 255, 0.03);
            border: 1px solid var(--border);
            border-radius: 20px;
            font-size: 0.75rem;
            color: var(--text-muted);
        }

        .badge svg {
            width: 14px; height: 14px;
            color: var(--success);
        }

        .badge.new-format {
            border-color: rgba(99, 102, 241, 0.3);
            background: rgba(99, 102, 241, 0.1);
        }

        /* Card */
        .card {
            background: var(--bg-card);
            border: 1px solid var(--border);
            border-radius: 24px;
            padding: 32px;
            backdrop-filter: blur(20px);
            position: relative;
            overflow: hidden;
        }

        .card::before {
            content: '';
            position: absolute;
            top: 0; left: 0; right: 0;
            height: 1px;
            background: linear-gradient(90deg, transparent, rgba(255,255,255,0.1), transparent);
        }

        /* Format indicator */
        .format-support {
            display: flex;
            gap: 12px;
            margin-bottom: 24px;
            padding: 16px;
            background: rgba(99, 102, 241, 0.05);
            border: 1px solid rgba(99, 102, 241, 0.1);
            border-radius: 12px;
        }

        .format-item {
            flex: 1;
            text-align: center;
            padding: 12px;
            border-radius: 8px;
            background: rgba(255, 255, 255, 0.03);
        }

        .format-item svg {
            width: 32px; height: 32px;
            margin-bottom: 8px;
            color: var(--primary);
        }

        .format-item .label {
            font-size: 0.7rem;
            color: var(--text-muted);
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .format-item .name {
            font-size: 0.85rem;
            font-weight: 500;
            margin-top: 4px;
        }

        /* Upload zone */
        .upload-zone {
            border: 2px dashed var(--border);
            border-radius: 16px;
            padding: 48px 24px;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        .upload-zone::before {
            content: '';
            position: absolute;
            inset: 0;
            background: linear-gradient(135deg, rgba(99, 102, 241, 0.05), transparent);
            opacity: 0;
            transition: opacity 0.3s;
        }

        .upload-zone:hover::before,
        .upload-zone.dragover::before {
            opacity: 1;
        }

        .upload-zone:hover,
        .upload-zone.dragover {
            border-color: var(--primary);
            transform: translateY(-2px);
        }

        .upload-icon {
            width: 64px; height: 64px;
            margin: 0 auto 20px;
            background: linear-gradient(135deg, rgba(99, 102, 241, 0.2), rgba(139, 92, 246, 0.2));
            border-radius: 16px;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .upload-icon svg {
            width: 32px; height: 32px;
            color: var(--primary);
        }

        .upload-zone h3 {
            font-size: 1rem;
            font-weight: 500;
            margin-bottom: 8px;
        }

        .upload-zone p {
            font-size: 0.85rem;
            color: var(--text-muted);
        }

        .upload-formats {
            display: flex;
            justify-content: center;
            gap: 8px;
            margin-top: 16px;
        }

        .format-tag {
            padding: 4px 10px;
            background: rgba(255, 255, 255, 0.05);
            border-radius: 6px;
            font-size: 0.7rem;
            font-family: 'JetBrains Mono', monospace;
            color: var(--text-muted);
        }

        input[type="file"] { display: none; }

        /* Preview */
        .preview {
            margin: 24px 0;
            text-align: center;
            display: none;
        }

        .preview.active { display: block; }

        .preview-container {
            position: relative;
            display: inline-block;
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 20px 40px -10px rgba(0, 0, 0, 0.5);
        }

        .preview img {
            max-width: 100%;
            max-height: 180px;
            display: block;
        }

        .preview-overlay {
            position: absolute;
            inset: 0;
            background: linear-gradient(to top, rgba(0,0,0,0.5), transparent);
            display: flex;
            align-items: flex-end;
            padding: 12px;
        }

        .preview-overlay span {
            font-size: 0.75rem;
            color: white;
            background: rgba(0,0,0,0.5);
            padding: 4px 8px;
            border-radius: 4px;
        }

        /* Button */
        .btn {
            width: 100%;
            padding: 16px 24px;
            border: none;
            border-radius: 12px;
            font-family: 'Inter', sans-serif;
            font-size: 1rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
            margin-top: 24px;
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--primary), #8b5cf6);
            color: white;
            box-shadow: 0 10px 30px -10px var(--primary-glow);
        }

        .btn-primary:hover:not(:disabled) {
            transform: translateY(-2px);
            box-shadow: 0 15px 40px -10px var(--primary-glow);
        }

        .btn:disabled {
            opacity: 0.5;
            cursor: not-allowed;
            transform: none;
        }

        .btn-content {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
        }

        /* LOADING ANIMATION */
        .loading-overlay {
            display: none;
            position: fixed;
            inset: 0;
            background: rgba(10, 10, 15, 0.95);
            z-index: 1000;
            backdrop-filter: blur(10px);
        }

        .loading-overlay.active {
            display: flex;
        }

        .loading-container {
            width: 100%;
            max-width: 400px;
            margin: auto;
            padding: 20px;
            text-align: center;
        }

        /* 3D Card Animation */
        .card-3d-container {
            perspective: 1000px;
            width: 280px;
            height: 180px;
            margin: 0 auto 40px;
        }

        .card-3d {
            width: 100%;
            height: 100%;
            position: relative;
            transform-style: preserve-3d;
            animation: cardFloat 3s ease-in-out infinite;
        }

        @keyframes cardFloat {
            0%, 100% { transform: translateY(0) rotateX(10deg) rotateY(-10deg); }
            50% { transform: translateY(-20px) rotateX(10deg) rotateY(10deg); }
        }

        .card-3d-face {
            position: absolute;
            width: 100%;
            height: 100%;
            border-radius: 12px;
            backface-visibility: hidden;
            display: flex;
            flex-direction: column;
            padding: 16px;
            box-shadow: 
                0 30px 60px -20px rgba(0, 0, 0, 0.5),
                0 0 40px -10px var(--primary-glow);
        }

        .card-front {
            background: linear-gradient(135deg, #1e3a5f 0%, #0d1b2a 50%, #1e3a5f 100%);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        /* Nouveau format - style carte bancaire */
        .card-new-format {
            background: linear-gradient(135deg, #1a237e 0%, #0d47a1 50%, #1565c0 100%);
        }

        .card-header-row {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 8px;
        }

        .card-title-section {
            flex: 1;
        }

        .card-title {
            font-size: 0.5rem;
            font-weight: 600;
            letter-spacing: 1px;
            color: rgba(255,255,255,0.9);
            text-transform: uppercase;
        }

        .card-subtitle {
            font-size: 0.45rem;
            color: rgba(255,255,255,0.6);
            margin-top: 2px;
        }

        .card-flag {
            display: flex;
            width: 28px;
            height: 20px;
            border-radius: 2px;
            overflow: hidden;
            box-shadow: 0 2px 4px rgba(0,0,0,0.2);
        }

        .card-flag span { flex: 1; }
        .card-flag .blue { background: #002395; }
        .card-flag .white { background: #ffffff; }
        .card-flag .red { background: #ed2939; }

        .eu-stars {
            width: 20px;
            height: 20px;
            background: #003399;
            border-radius: 50%;
            margin-left: 6px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 8px;
            color: #ffcc00;
        }

        .card-body {
            display: flex;
            gap: 12px;
            flex: 1;
            align-items: flex-start;
        }

        .card-photo {
            width: 50px;
            height: 62px;
            background: linear-gradient(135deg, #374151, #1f2937);
            border-radius: 4px;
            position: relative;
            overflow: hidden;
            flex-shrink: 0;
        }

        .card-photo::before {
            content: '';
            position: absolute;
            top: 20%;
            left: 50%;
            transform: translateX(-50%);
            width: 20px;
            height: 20px;
            background: #4b5563;
            border-radius: 50%;
        }

        .card-photo::after {
            content: '';
            position: absolute;
            bottom: 15%;
            left: 50%;
            transform: translateX(-50%);
            width: 28px;
            height: 16px;
            background: #4b5563;
            border-radius: 50% 50% 0 0;
        }

        .card-info {
            flex: 1;
        }

        .card-info-line {
            height: 6px;
            background: rgba(255,255,255,0.15);
            border-radius: 3px;
            margin-bottom: 5px;
            position: relative;
            overflow: hidden;
        }

        .card-info-line::after {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
            animation: shimmer 2s infinite;
        }

        .card-info-line:nth-child(2) { width: 75%; }
        .card-info-line:nth-child(3) { width: 60%; }
        .card-info-line:nth-child(4) { width: 85%; }

        @keyframes shimmer {
            100% { left: 100%; }
        }

        .card-chip {
            width: 36px;
            height: 28px;
            background: linear-gradient(135deg, #fbbf24, #f59e0b);
            border-radius: 4px;
            position: absolute;
            bottom: 40px;
            left: 16px;
        }

        .card-chip::before {
            content: '';
            position: absolute;
            inset: 3px;
            background: linear-gradient(135deg, transparent 30%, rgba(255,255,255,0.3) 50%, transparent 70%);
        }

        .card-mrz {
            position: absolute;
            bottom: 8px;
            left: 8px;
            right: 8px;
            font-family: 'JetBrains Mono', monospace;
            font-size: 6px;
            color: rgba(255,255,255,0.4);
            line-height: 1.4;
            letter-spacing: 1px;
        }

        .card-mrz.new-format {
            font-size: 5px;
        }

        .hologram {
            position: absolute;
            bottom: 35px;
            right: 12px;
            width: 30px;
            height: 30px;
            border-radius: 50%;
            background: conic-gradient(
                from 0deg,
                #ff0000, #ff8000, #ffff00, #80ff00,
                #00ff00, #00ff80, #00ffff, #0080ff,
                #0000ff, #8000ff, #ff00ff, #ff0080, #ff0000
            );
            opacity: 0.5;
            animation: hologramRotate 3s linear infinite;
        }

        @keyframes hologramRotate {
            100% { transform: rotate(360deg); }
        }

        .hologram::before {
            content: '';
            position: absolute;
            inset: 4px;
            background: #1a237e;
            border-radius: 50%;
        }

        .scan-line {
            position: absolute;
            left: 0;
            right: 0;
            height: 3px;
            background: linear-gradient(90deg, transparent, var(--success), transparent);
            box-shadow: 0 0 20px var(--success), 0 0 40px var(--success);
            animation: scan 2s ease-in-out infinite;
        }

        @keyframes scan {
            0%, 100% { top: 10%; opacity: 1; }
            50% { top: 90%; opacity: 1; }
        }

        /* Progress steps */
        .progress-steps {
            display: flex;
            flex-direction: column;
            gap: 12px;
            margin-bottom: 32px;
        }

        .step {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 12px;
            background: rgba(255, 255, 255, 0.03);
            border: 1px solid var(--border);
            border-radius: 10px;
            transition: all 0.5s ease;
        }

        .step.active {
            background: rgba(99, 102, 241, 0.1);
            border-color: var(--primary);
        }

        .step.completed {
            background: rgba(16, 185, 129, 0.1);
            border-color: var(--success);
        }

        .step-icon {
            width: 36px;
            height: 36px;
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            background: rgba(255, 255, 255, 0.05);
            position: relative;
            flex-shrink: 0;
        }

        .step.active .step-icon { background: var(--primary); }
        .step.completed .step-icon { background: var(--success); }

        .step-icon svg {
            width: 18px;
            height: 18px;
            color: var(--text-muted);
        }

        .step.active .step-icon svg,
        .step.completed .step-icon svg {
            color: white;
        }

        .step-spinner {
            position: absolute;
            inset: -3px;
            border: 2px solid transparent;
            border-top-color: var(--primary);
            border-radius: 10px;
            animation: spin 1s linear infinite;
            display: none;
        }

        .step.active .step-spinner { display: block; }

        @keyframes spin {
            100% { transform: rotate(360deg); }
        }

        .step-content { flex: 1; min-width: 0; }

        .step-title {
            font-weight: 500;
            font-size: 0.85rem;
            margin-bottom: 2px;
        }

        .step-desc {
            font-size: 0.7rem;
            color: var(--text-muted);
        }

        .step-status {
            font-family: 'JetBrains Mono', monospace;
            font-size: 0.65rem;
            color: var(--text-muted);
            flex-shrink: 0;
        }

        .step.active .step-status { color: var(--primary); }
        .step.completed .step-status { color: var(--success); }

        /* Percentage */
        .percentage {
            font-family: 'JetBrains Mono', monospace;
            font-size: 2.5rem;
            font-weight: 700;
            background: linear-gradient(135deg, var(--primary), var(--success));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
            margin-bottom: 8px;
        }

        .loading-text {
            font-size: 0.85rem;
            color: var(--text-muted);
        }

        /* Binary rain */
        .binary-rain {
            position: absolute;
            top: 0; left: 0; right: 0; bottom: 0;
            overflow: hidden;
            pointer-events: none;
            opacity: 0.1;
        }

        .binary-column {
            position: absolute;
            top: -100%;
            font-family: 'JetBrains Mono', monospace;
            font-size: 12px;
            color: var(--success);
            writing-mode: vertical-rl;
            animation: binaryFall linear infinite;
        }

        @keyframes binaryFall {
            0% { transform: translateY(0); }
            100% { transform: translateY(200vh); }
        }

        /* Result card - Animated ID Card */
        .result-card-container {
            margin-top: 32px;
            text-align: center;
        }

        .result-card {
            width: 100%;
            max-width: 380px;
            aspect-ratio: 1.586;
            margin: 0 auto 24px;
            perspective: 1000px;
            animation: cardAppear 0.8s ease-out;
        }

        @keyframes cardAppear {
            0% {
                opacity: 0;
                transform: scale(0.8) rotateX(20deg);
            }
            100% {
                opacity: 1;
                transform: scale(1) rotateX(0);
            }
        }

        .result-card-inner {
            width: 100%;
            height: 100%;
            background: linear-gradient(135deg, #1e3a5f 0%, #0d1b2a 40%, #1e3a5f 100%);
            border-radius: 16px;
            padding: 16px;
            position: relative;
            overflow: hidden;
            box-shadow: 
                0 25px 50px -12px rgba(0, 0, 0, 0.5),
                0 0 0 1px rgba(255, 255, 255, 0.1),
                inset 0 1px 0 rgba(255, 255, 255, 0.1);
            display: flex;
            flex-direction: column;
        }

        .result-card.adult .result-card-inner {
            background: linear-gradient(135deg, #064e3b 0%, #022c22 40%, #064e3b 100%);
        }

        .result-card.minor .result-card-inner {
            background: linear-gradient(135deg, #7f1d1d 0%, #450a0a 40%, #7f1d1d 100%);
        }

        /* Card Header */
        .rc-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 12px;
        }

        .rc-header-left {
            display: flex;
            align-items: flex-start;
            gap: 10px;
        }

        .rc-flag {
            display: flex;
            width: 24px;
            height: 18px;
            border-radius: 2px;
            overflow: hidden;
            box-shadow: 0 2px 4px rgba(0,0,0,0.3);
            flex-shrink: 0;
        }

        .rc-flag span { flex: 1; }
        .rc-flag .blue { background: #002395; }
        .rc-flag .white { background: #ffffff; }
        .rc-flag .red { background: #ed2939; }

        .rc-titles {
            display: flex;
            flex-direction: column;
        }

        .rc-title {
            font-size: 0.6rem;
            font-weight: 700;
            letter-spacing: 1px;
            color: rgba(255,255,255,0.95);
            animation: typeIn 0.5s ease-out 0.3s both;
        }

        .rc-subtitle {
            font-size: 0.45rem;
            color: rgba(255,255,255,0.6);
            letter-spacing: 0.5px;
            margin-top: 2px;
            animation: typeIn 0.5s ease-out 0.5s both;
        }

        @keyframes typeIn {
            from {
                opacity: 0;
                transform: translateX(-10px);
            }
            to {
                opacity: 1;
                transform: translateX(0);
            }
        }

        .rc-header-right {
            display: flex;
            align-items: center;
        }

        .rc-eu {
            background: #003399;
            border-radius: 4px;
            padding: 4px 8px;
            display: flex;
            flex-direction: column;
            align-items: center;
            animation: fadeIn 0.5s ease-out 0.4s both;
        }

        .rc-eu-stars {
            font-size: 4px;
            color: #ffcc00;
            letter-spacing: -1px;
            line-height: 1;
        }

        .rc-eu span {
            font-size: 0.6rem;
            font-weight: 700;
            color: white;
            margin-top: 2px;
        }

        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }

        /* Card Body */
        .rc-body {
            display: flex;
            gap: 16px;
            flex: 1;
        }

        .rc-photo {
            width: 80px;
            height: 100px;
            background: linear-gradient(135deg, #1f2937 0%, #111827 100%);
            border-radius: 6px;
            flex-shrink: 0;
            position: relative;
            overflow: hidden;
            box-shadow: inset 0 2px 4px rgba(0,0,0,0.3);
            animation: photoAppear 0.6s ease-out 0.6s both;
        }

        @keyframes photoAppear {
            from {
                opacity: 0;
                transform: scale(0.8);
            }
            to {
                opacity: 1;
                transform: scale(1);
            }
        }

        .rc-photo-placeholder {
            position: absolute;
            inset: 0;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .rc-photo-placeholder svg {
            width: 40px;
            height: 40px;
            color: #4b5563;
        }

        .rc-sexe {
            position: absolute;
            bottom: 4px;
            right: 4px;
            font-size: 1rem;
            opacity: 0.7;
        }

        .rc-info {
            flex: 1;
            display: flex;
            flex-direction: column;
            gap: 8px;
        }

        .rc-field {
            display: flex;
            flex-direction: column;
            opacity: 0;
            animation: fieldAppear 0.5s ease-out forwards;
            animation-delay: var(--delay, 0s);
        }

        @keyframes fieldAppear {
            from {
                opacity: 0;
                transform: translateY(10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .rc-label {
            font-size: 0.5rem;
            color: rgba(255,255,255,0.5);
            text-transform: uppercase;
            letter-spacing: 0.5px;
            margin-bottom: 2px;
        }

        .rc-value {
            font-size: 0.85rem;
            font-weight: 600;
            color: white;
            font-family: 'JetBrains Mono', monospace;
        }

        .rc-date {
            color: #60a5fa;
        }

        .result-card.adult .rc-date {
            color: #34d399;
        }

        .result-card.minor .rc-date {
            color: #f87171;
        }

        .rc-lieu {
            font-size: 0.7rem;
        }

        /* Card Footer */
        .rc-footer {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-top: auto;
            padding-top: 12px;
        }

        .rc-chip {
            width: 40px;
            height: 30px;
            background: linear-gradient(135deg, #fbbf24, #f59e0b, #d97706);
            border-radius: 5px;
            position: relative;
            animation: chipAppear 0.5s ease-out 1s both;
        }

        @keyframes chipAppear {
            from {
                opacity: 0;
                transform: scale(0);
            }
            to {
                opacity: 1;
                transform: scale(1);
            }
        }

        .rc-chip::before {
            content: '';
            position: absolute;
            top: 50%;
            left: 4px;
            right: 4px;
            height: 1px;
            background: rgba(0,0,0,0.2);
        }

        .rc-chip::after {
            content: '';
            position: absolute;
            top: 4px;
            bottom: 4px;
            left: 50%;
            width: 1px;
            background: rgba(0,0,0,0.2);
        }

        .rc-status {
            display: flex;
            align-items: center;
            gap: 10px;
            padding: 8px 16px;
            border-radius: 8px;
            animation: statusAppear 0.6s ease-out 1.2s both;
        }

        @keyframes statusAppear {
            from {
                opacity: 0;
                transform: scale(0.8);
            }
            to {
                opacity: 1;
                transform: scale(1);
            }
        }

        .rc-status.adult {
            background: rgba(16, 185, 129, 0.2);
            border: 1px solid rgba(16, 185, 129, 0.4);
        }

        .rc-status.minor {
            background: rgba(239, 68, 68, 0.2);
            border: 1px solid rgba(239, 68, 68, 0.4);
        }

        .rc-status-icon {
            font-size: 1.2rem;
            animation: iconPulse 1s ease-in-out infinite;
            animation-delay: 1.5s;
        }

        @keyframes iconPulse {
            0%, 100% { transform: scale(1); }
            50% { transform: scale(1.2); }
        }

        .rc-status-text {
            display: flex;
            flex-direction: column;
        }

        .rc-status-label {
            font-size: 0.75rem;
            font-weight: 700;
            letter-spacing: 1px;
        }

        .rc-status.adult .rc-status-label { color: #34d399; }
        .rc-status.minor .rc-status-label { color: #f87171; }

        .rc-status-age {
            font-size: 0.65rem;
            color: rgba(255,255,255,0.7);
        }

        .rc-hologram {
            width: 35px;
            height: 35px;
            border-radius: 50%;
            background: conic-gradient(
                from 0deg,
                #ff0000, #ff8000, #ffff00, #80ff00,
                #00ff00, #00ff80, #00ffff, #0080ff,
                #0000ff, #8000ff, #ff00ff, #ff0080, #ff0000
            );
            opacity: 0.5;
            animation: hologramSpin 4s linear infinite, fadeIn 0.5s ease-out 1.4s both;
            position: relative;
        }

        @keyframes hologramSpin {
            100% { transform: rotate(360deg); }
        }

        .rc-hologram::before {
            content: '';
            position: absolute;
            inset: 5px;
            background: inherit;
            border-radius: 50%;
            filter: blur(2px);
        }

        /* Card Effects */
        .rc-shine {
            position: absolute;
            top: 0;
            left: -100%;
            width: 50%;
            height: 100%;
            background: linear-gradient(
                90deg,
                transparent,
                rgba(255, 255, 255, 0.1),
                transparent
            );
            animation: shine 3s ease-in-out infinite;
            animation-delay: 2s;
        }

        @keyframes shine {
            0% { left: -100%; }
            50%, 100% { left: 200%; }
        }

        .rc-scan-complete {
            position: absolute;
            inset: 0;
            border: 2px solid transparent;
            border-radius: 16px;
            animation: scanComplete 1s ease-out 1.5s both;
        }

        .result-card.adult .rc-scan-complete {
            animation-name: scanCompleteAdult;
        }

        .result-card.minor .rc-scan-complete {
            animation-name: scanCompleteMinor;
        }

        @keyframes scanCompleteAdult {
            0% {
                border-color: transparent;
                box-shadow: none;
            }
            50% {
                border-color: #10b981;
                box-shadow: 0 0 20px rgba(16, 185, 129, 0.5), inset 0 0 20px rgba(16, 185, 129, 0.1);
            }
            100% {
                border-color: rgba(16, 185, 129, 0.3);
                box-shadow: none;
            }
        }

        @keyframes scanCompleteMinor {
            0% {
                border-color: transparent;
                box-shadow: none;
            }
            50% {
                border-color: #ef4444;
                box-shadow: 0 0 20px rgba(239, 68, 68, 0.5), inset 0 0 20px rgba(239, 68, 68, 0.1);
            }
            100% {
                border-color: rgba(239, 68, 68, 0.3);
                box-shadow: none;
            }
        }

        /* Result Message */
        .result-message {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
            padding: 16px 24px;
            border-radius: 12px;
            margin-bottom: 16px;
            animation: messageAppear 0.5s ease-out 1.8s both;
        }

        @keyframes messageAppear {
            from {
                opacity: 0;
                transform: translateY(10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .result-message.adult {
            background: linear-gradient(135deg, rgba(16, 185, 129, 0.15), rgba(16, 185, 129, 0.05));
            border: 1px solid rgba(16, 185, 129, 0.3);
        }

        .result-message.minor {
            background: linear-gradient(135deg, rgba(239, 68, 68, 0.15), rgba(239, 68, 68, 0.05));
            border: 1px solid rgba(239, 68, 68, 0.3);
        }

        .result-message-icon {
            font-size: 1.5rem;
        }

        .result-message-text {
            font-weight: 500;
        }

        .result-message.adult .result-message-text { color: #34d399; }
        .result-message.minor .result-message-text { color: #f87171; }

        .result-format {
            font-size: 0.75rem;
            color: var(--text-muted);
            animation: fadeIn 0.5s ease-out 2s both;
        }

        /* Error result */
        .result.error {
            margin-top: 24px;
            padding: 24px;
            border-radius: 16px;
            text-align: center;
            background: linear-gradient(135deg, rgba(245, 158, 11, 0.1), rgba(245, 158, 11, 0.05));
            border: 1px solid rgba(245, 158, 11, 0.3);
            animation: resultAppear 0.5s ease;
        }

        @keyframes resultAppear {
            from { opacity: 0; transform: translateY(20px); }
        }

        .result.error .result-icon {
            width: 72px;
            height: 72px;
            margin: 0 auto 16px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 2rem;
            background: linear-gradient(135deg, rgba(245, 158, 11, 0.2), rgba(245, 158, 11, 0.1));
        }

        .result.error h2 {
            font-size: 1.25rem;
            font-weight: 600;
            margin-bottom: 12px;
            color: var(--warning);
        }

        .result.error p {
            color: var(--text-muted);
        }

        /* Debug panel */
        .debug {
            margin-top: 24px;
            padding: 16px;
            background: rgba(0, 0, 0, 0.3);
            border: 1px solid var(--border);
            border-radius: 12px;
            font-family: 'JetBrains Mono', monospace;
            font-size: 0.65rem;
            max-height: 300px;
            overflow-y: auto;
        }

        .debug-section {
            margin-bottom: 12px;
            padding-bottom: 12px;
            border-bottom: 1px solid var(--border);
        }

        .debug-section:last-child { border-bottom: none; }

        .debug-title {
            color: var(--primary);
            font-weight: 600;
            margin-bottom: 6px;
        }

        /* Info box */
        .info-box {
            margin-top: 24px;
            padding: 16px;
            background: rgba(99, 102, 241, 0.05);
            border: 1px solid rgba(99, 102, 241, 0.1);
            border-radius: 12px;
        }

        .info-box h4 {
            font-size: 0.85rem;
            font-weight: 500;
            margin-bottom: 12px;
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .info-box ul {
            list-style: none;
            padding: 0;
        }

        .info-box li {
            font-size: 0.8rem;
            color: var(--text-muted);
            padding: 5px 0;
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .info-box li::before {
            content: '→';
            color: var(--primary);
        }
    </style>
</head>
<body>
    <div class="bg-animation"></div>
    <div class="grid-overlay"></div>

    <div class="container">
        <header class="header">
            <div class="logo">
                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z" />
                </svg>
            </div>
            <h1>Vérification d'identité</h1>
            <p>Analyse sécurisée de votre carte d'identité française</p>
            <div class="security-badges">
                <div class="badge">
                    <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z" />
                    </svg>
                    Chiffré
                </div>
                <div class="badge">
                    <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7" />
                    </svg>
                    RGPD
                </div>
                <div class="badge new-format">
                    <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4M7.835 4.697a3.42 3.42 0 001.946-.806 3.42 3.42 0 014.438 0 3.42 3.42 0 001.946.806 3.42 3.42 0 013.138 3.138 3.42 3.42 0 00.806 1.946 3.42 3.42 0 010 4.438 3.42 3.42 0 00-.806 1.946 3.42 3.42 0 01-3.138 3.138 3.42 3.42 0 00-1.946.806 3.42 3.42 0 01-4.438 0 3.42 3.42 0 00-1.946-.806 3.42 3.42 0 01-3.138-3.138 3.42 3.42 0 00-.806-1.946 3.42 3.42 0 010-4.438 3.42 3.42 0 00.806-1.946 3.42 3.42 0 013.138-3.138z" />
                    </svg>
                    Format 2021+
                </div>
            </div>
        </header>

        <div class="card">
            <!-- Format support indicator -->
            <div class="format-support">
                <div class="format-item">
                    <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 10h18M7 15h1m4 0h1m-7 4h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 00-3 3v8a3 3 0 003 3z" />
                    </svg>
                    <div class="label">Nouveau</div>
                    <div class="name">Format 2021+</div>
                </div>
                <div class="format-item">
                    <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 6H5a2 2 0 00-2 2v9a2 2 0 002 2h14a2 2 0 002-2V8a2 2 0 00-2-2h-5m-4 0V5a2 2 0 114 0v1m-4 0a2 2 0 104 0m-5 8a2 2 0 100-4 2 2 0 000 4zm0 0c1.306 0 2.417.835 2.83 2M9 14a3.001 3.001 0 00-2.83 2M15 11h3m-3 4h2" />
                    </svg>
                    <div class="label">Ancien</div>
                    <div class="name">Format pré-2021</div>
                </div>
            </div>

            <form method="POST" enctype="multipart/form-data" id="uploadForm">
                <div class="upload-zone" id="dropZone">
                    <div class="upload-icon">
                        <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
                        </svg>
                    </div>
                    <h3>Déposez votre carte d'identité</h3>
                    <p>Recto avec la date de naissance visible</p>
                    <div class="upload-formats">
                        <span class="format-tag">JPG</span>
                        <span class="format-tag">PNG</span>
                        <span class="format-tag">WEBP</span>
                        <span class="format-tag">BMP</span>
                    </div>
                </div>
                <input type="file" name="id_card" id="fileInput" accept="image/*">

                <div class="preview" id="preview">
                    <div class="preview-container">
                        <img id="previewImg" src="" alt="Aperçu">
                        <div class="preview-overlay">
                            <span id="fileName">document.jpg</span>
                        </div>
                    </div>
                </div>

                <button type="submit" class="btn btn-primary" id="submitBtn" disabled>
                    <span class="btn-content">
                        <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z" />
                        </svg>
                        Lancer la vérification
                    </span>
                </button>
            </form>

            <?php if ($result): ?>
                <?php if ($result['success']): ?>
                    <!-- CARTE D'IDENTITÉ ANIMÉE -->
                    <div class="result-card-container">
                        <div class="result-card <?= $result['is_adult'] ? 'adult' : 'minor' ?>" id="resultCard">
                            <div class="result-card-inner">
                                <!-- Header -->
                                <div class="rc-header">
                                    <div class="rc-header-left">
                                        <div class="rc-flag">
                                            <span class="blue"></span>
                                            <span class="white"></span>
                                            <span class="red"></span>
                                        </div>
                                        <div class="rc-titles">
                                            <div class="rc-title">RÉPUBLIQUE FRANÇAISE</div>
                                            <div class="rc-subtitle">CARTE NATIONALE D'IDENTITÉ</div>
                                        </div>
                                    </div>
                                    <div class="rc-header-right">
                                        <div class="rc-eu">
                                            <div class="rc-eu-stars">★★★★★★★★★★★★</div>
                                            <span>FR</span>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Body -->
                                <div class="rc-body">
                                    <div class="rc-photo">
                                        <div class="rc-photo-placeholder">
                                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z" />
                                            </svg>
                                        </div>
                                        <?php if ($result['sexe']): ?>
                                            <div class="rc-sexe"><?= $result['sexe'] === 'F' ? '♀' : '♂' ?></div>
                                        <?php endif; ?>
                                    </div>
                                    <div class="rc-info">
                                        <div class="rc-field animate-field" style="--delay: 0.2s">
                                            <span class="rc-label">Nom</span>
                                            <span class="rc-value" id="rcNom"><?= htmlspecialchars($result['nom'] ?? '—') ?></span>
                                        </div>
                                        <div class="rc-field animate-field" style="--delay: 0.4s">
                                            <span class="rc-label">Prénom</span>
                                            <span class="rc-value" id="rcPrenom"><?= htmlspecialchars($result['prenom'] ?? '—') ?></span>
                                        </div>
                                        <div class="rc-field animate-field" style="--delay: 0.6s">
                                            <span class="rc-label">Date de naissance</span>
                                            <span class="rc-value rc-date" id="rcDate"><?= date('d/m/Y', strtotime($result['birth_date'])) ?></span>
                                        </div>
                                        <?php if ($result['lieu_naissance']): ?>
                                        <div class="rc-field animate-field" style="--delay: 0.8s">
                                            <span class="rc-label">Lieu de naissance</span>
                                            <span class="rc-value rc-lieu"><?= htmlspecialchars($result['lieu_naissance']) ?></span>
                                        </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                
                                <!-- Footer avec statut -->
                                <div class="rc-footer">
                                    <div class="rc-chip"></div>
                                    <div class="rc-status <?= $result['is_adult'] ? 'adult' : 'minor' ?>">
                                        <div class="rc-status-icon"><?= $result['is_adult'] ? '✓' : '✗' ?></div>
                                        <div class="rc-status-text">
                                            <span class="rc-status-label"><?= $result['is_adult'] ? 'MAJEUR' : 'MINEUR' ?></span>
                                            <span class="rc-status-age"><?= $result['age'] ?> ans</span>
                                        </div>
                                    </div>
                                    <div class="rc-hologram"></div>
                                </div>
                                
                                <!-- Effets -->
                                <div class="rc-shine"></div>
                                <div class="rc-scan-complete"></div>
                            </div>
                        </div>
                        
                        <!-- Message sous la carte -->
                        <div class="result-message <?= $result['is_adult'] ? 'adult' : 'minor' ?>">
                            <span class="result-message-icon"><?= $result['is_adult'] ? '🎉' : '🚫' ?></span>
                            <span class="result-message-text">
                                <?= $result['is_adult'] ? 'Vérification réussie - Accès autorisé' : 'Vérification réussie - Accès refusé (mineur)' ?>
                            </span>
                        </div>
                        
                        <div class="result-format">
                            Format détecté : <?= $result['format'] === 'new' ? 'Nouveau (2021+)' : ($result['format'] === 'old' ? 'Ancien (pré-2021)' : 'Non déterminé') ?>
                        </div>
                    </div>
                <?php else: ?>
                    <div class="result error">
                        <div class="result-icon">⚠️</div>
                        <h2>Vérification impossible</h2>
                        <p><?= htmlspecialchars($result['message']) ?></p>
                    </div>
                <?php endif; ?>

                <?php if ($showDebug && !empty($result['debug'])): ?>
                    <div class="debug">
                        <?php if (isset($result['debug']['detected_format'])): ?>
                            <div class="debug-section">
                                <div class="debug-title">📋 Format détecté</div>
                                <?= $result['debug']['detected_format'] ?>
                            </div>
                        <?php endif; ?>
                        <?php if (isset($result['debug']['extraction_method'])): ?>
                            <div class="debug-section">
                                <div class="debug-title">🔍 Méthode d'extraction</div>
                                <?= $result['debug']['extraction_method'] ?>
                            </div>
                        <?php endif; ?>
                        <?php if (isset($result['debug']['rotation_scores'])): ?>
                            <div class="debug-section">
                                <div class="debug-title">🔄 Scores par rotation</div>
                                <?php foreach ($result['debug']['rotation_scores'] as $rot => $data): ?>
                                    <div><?= $rot ?>° : score=<?= $data['score'] ?> (<?= $data['format'] ?? '?' ?>)</div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                        <?php if (isset($result['debug']['mrz_pattern'])): ?>
                            <div class="debug-section">
                                <div class="debug-title">🎯 Pattern MRZ</div>
                                <?= htmlspecialchars($result['debug']['mrz_pattern']) ?>
                            </div>
                        <?php endif; ?>
                        <?php if (isset($result['debug']['visual_pattern_matched'])): ?>
                            <div class="debug-section">
                                <div class="debug-title">👁️ Pattern visuel</div>
                                <?= $result['debug']['visual_pattern_matched'] ?>
                            </div>
                        <?php endif; ?>
                        <?php if (isset($result['debug']['cleaned_text'])): ?>
                            <div class="debug-section">
                                <div class="debug-title">📝 Texte extrait</div>
                                <?= htmlspecialchars(substr($result['debug']['cleaned_text'], 0, 400)) ?>
                            </div>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            <?php endif; ?>

            <div class="info-box">
                <h4>
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                    </svg>
                    Pour un résultat optimal
                </h4>
                <ul>
                    <li>Photo nette et bien éclairée</li>
                    <li>Recto de la carte visible (avec date de naissance)</li>
                    <li>Évitez les reflets sur la puce ou l'hologramme</li>
                    <li>Toutes orientations acceptées (rotation auto)</li>
                    <li>Ancien et nouveau format CNI supportés</li>
                </ul>
            </div>
        </div>
    </div>

    <!-- LOADING OVERLAY -->
    <div class="loading-overlay" id="loadingOverlay">
        <div class="binary-rain" id="binaryRain"></div>
        <div class="loading-container">
            <div class="card-3d-container">
                <div class="card-3d" id="card3d">
                    <div class="card-3d-face card-front card-new-format">
                        <div class="card-header-row">
                            <div class="card-title-section">
                                <div class="card-title">République Française</div>
                                <div class="card-subtitle">Carte Nationale d'Identité</div>
                            </div>
                            <div style="display: flex; align-items: center;">
                                <div class="card-flag">
                                    <span class="blue"></span>
                                    <span class="white"></span>
                                    <span class="red"></span>
                                </div>
                                <div class="eu-stars">★</div>
                            </div>
                        </div>
                        <div class="card-body">
                            <div class="card-photo"></div>
                            <div class="card-info">
                                <div class="card-info-line"></div>
                                <div class="card-info-line"></div>
                                <div class="card-info-line"></div>
                                <div class="card-info-line"></div>
                            </div>
                        </div>
                        <div class="card-chip"></div>
                        <div class="hologram"></div>
                        <div class="card-mrz new-format">
                            IDFRA&lt;&lt;&lt;&lt;&lt;&lt;&lt;&lt;&lt;&lt;&lt;&lt;&lt;&lt;&lt;&lt;&lt;&lt;&lt;&lt;&lt;&lt;&lt;&lt;<br>
                            0000000000FRA0000000M&lt;&lt;&lt;&lt;&lt;&lt;&lt;&lt;<br>
                            SPECIMEN&lt;&lt;JEAN&lt;&lt;&lt;&lt;&lt;&lt;&lt;&lt;&lt;&lt;&lt;&lt;
                        </div>
                        <div class="scan-line"></div>
                    </div>
                </div>
            </div>

            <div class="percentage" id="percentage">0%</div>
            <div class="loading-text" id="loadingText">Initialisation...</div>

            <div class="progress-steps">
                <div class="step" id="step1">
                    <div class="step-icon">
                        <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
                        </svg>
                        <div class="step-spinner"></div>
                    </div>
                    <div class="step-content">
                        <div class="step-title">Chargement du document</div>
                        <div class="step-desc">Analyse de l'image</div>
                    </div>
                    <div class="step-status">En attente</div>
                </div>
                <div class="step" id="step2">
                    <div class="step-icon">
                        <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 21a4 4 0 01-4-4V5a2 2 0 012-2h4a2 2 0 012 2v12a4 4 0 01-4 4zm0 0h12a2 2 0 002-2v-4a2 2 0 00-2-2h-2.343M11 7.343l1.657-1.657a2 2 0 012.828 0l2.829 2.829a2 2 0 010 2.828l-8.486 8.485M7 17h.01" />
                        </svg>
                        <div class="step-spinner"></div>
                    </div>
                    <div class="step-content">
                        <div class="step-title">Détection du format</div>
                        <div class="step-desc">Nouveau ou ancien format CNI</div>
                    </div>
                    <div class="step-status">En attente</div>
                </div>
                <div class="step" id="step3">
                    <div class="step-icon">
                        <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2m-3 7h3m-3 4h3m-6-4h.01M9 16h.01" />
                        </svg>
                        <div class="step-spinner"></div>
                    </div>
                    <div class="step-content">
                        <div class="step-title">Extraction OCR + MRZ</div>
                        <div class="step-desc">Lecture des informations</div>
                    </div>
                    <div class="step-status">En attente</div>
                </div>
                <div class="step" id="step4">
                    <div class="step-icon">
                        <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
                        </svg>
                        <div class="step-spinner"></div>
                    </div>
                    <div class="step-content">
                        <div class="step-title">Vérification de l'âge</div>
                        <div class="step-desc">Calcul et validation</div>
                    </div>
                    <div class="step-status">En attente</div>
                </div>
            </div>
        </div>
    </div>

    <script>
        const dropZone = document.getElementById('dropZone');
        const fileInput = document.getElementById('fileInput');
        const preview = document.getElementById('preview');
        const previewImg = document.getElementById('previewImg');
        const fileName = document.getElementById('fileName');
        const submitBtn = document.getElementById('submitBtn');
        const form = document.getElementById('uploadForm');
        const loadingOverlay = document.getElementById('loadingOverlay');
        const percentage = document.getElementById('percentage');
        const loadingText = document.getElementById('loadingText');
        const binaryRain = document.getElementById('binaryRain');

        // Binary rain
        function createBinaryRain() {
            binaryRain.innerHTML = '';
            for (let i = 0; i < 15; i++) {
                const col = document.createElement('div');
                col.className = 'binary-column';
                col.style.left = `${(i / 15) * 100}%`;
                col.style.animationDuration = `${3 + Math.random() * 4}s`;
                col.style.animationDelay = `${Math.random() * 2}s`;
                let binary = '';
                for (let j = 0; j < 40; j++) binary += Math.random() > 0.5 ? '1' : '0';
                col.textContent = binary;
                binaryRain.appendChild(col);
            }
        }

        // Upload
        dropZone.addEventListener('click', () => fileInput.click());
        dropZone.addEventListener('dragover', (e) => { e.preventDefault(); dropZone.classList.add('dragover'); });
        dropZone.addEventListener('dragleave', () => dropZone.classList.remove('dragover'));
        dropZone.addEventListener('drop', (e) => {
            e.preventDefault();
            dropZone.classList.remove('dragover');
            if (e.dataTransfer.files.length) {
                fileInput.files = e.dataTransfer.files;
                handleFile(e.dataTransfer.files[0]);
            }
        });

        fileInput.addEventListener('change', (e) => {
            if (e.target.files.length) handleFile(e.target.files[0]);
        });

        function handleFile(file) {
            if (!file.type.startsWith('image/')) { alert('Veuillez sélectionner une image'); return; }
            const reader = new FileReader();
            reader.onload = (e) => {
                previewImg.src = e.target.result;
                fileName.textContent = file.name;
                preview.classList.add('active');
                submitBtn.disabled = false;
            };
            reader.readAsDataURL(file);
        }

        // Submit
        form.addEventListener('submit', () => {
            loadingOverlay.classList.add('active');
            createBinaryRain();

            const steps = ['step1', 'step2', 'step3', 'step4'];
            const messages = ['Chargement du document...', 'Détection du format CNI...', 'Extraction OCR + MRZ...', 'Vérification de l\'âge...'];
            let currentStep = 0;
            let currentPercent = 0;

            function updateStep(idx, status) {
                const step = document.getElementById(steps[idx]);
                const statusEl = step.querySelector('.step-status');
                if (status === 'active') {
                    step.classList.add('active');
                    step.classList.remove('completed');
                    statusEl.textContent = 'En cours...';
                } else if (status === 'completed') {
                    step.classList.remove('active');
                    step.classList.add('completed');
                    statusEl.textContent = '✓ Terminé';
                }
            }

            function animateProgress() {
                const target = Math.min((currentStep + 1) * 25, 95);
                if (currentPercent < target) {
                    currentPercent++;
                    percentage.textContent = currentPercent + '%';
                    requestAnimationFrame(animateProgress);
                }
            }

            function nextStep() {
                if (currentStep < steps.length) {
                    if (currentStep > 0) updateStep(currentStep - 1, 'completed');
                    updateStep(currentStep, 'active');
                    loadingText.textContent = messages[currentStep];
                    animateProgress();
                    currentStep++;
                    if (currentStep < steps.length) setTimeout(nextStep, 1200 + Math.random() * 800);
                }
            }

            setTimeout(nextStep, 400);
        });
    </script>
</body>
</html>

