<?php
require_once __DIR__ . '/config.php';

// Vérifier que le joueur est connecté
if (!isset($_SESSION['player_id']) || !isset($_SESSION['game_code'])) {
    header('Location: index.php');
    exit;
}

$playerId = $_SESSION['player_id'];
$gameCode = $_SESSION['game_code'];

// Charger la partie
$game = loadGame($gameCode);

if (!$game) {
    unset($_SESSION['player_id'], $_SESSION['game_code'], $_SESSION['player_name']);
    header('Location: index.php');
    exit;
}

// Vérifier que le joueur fait partie de la partie
if (!isset($game['players'][$playerId])) {
    unset($_SESSION['player_id'], $_SESSION['game_code'], $_SESSION['player_name']);
    header('Location: index.php');
    exit;
}

$player = $game['players'][$playerId];
$isHost = ($game['host'] === $playerId);
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>Undercover - Partie <?= htmlspecialchars($gameCode) ?></title>
    <style>
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif;
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 50%, #0f3460 100%);
            min-height: 100vh;
            color: #fff;
            padding: 15px;
            padding-bottom: 100px;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding: 15px;
            background: rgba(255, 255, 255, 0.05);
            border-radius: 15px;
            backdrop-filter: blur(10px);
        }
        
        .game-code {
            background: linear-gradient(45deg, #e94560, #ff6b6b);
            padding: 8px 15px;
            border-radius: 10px;
            font-weight: bold;
            font-size: 1.2rem;
            letter-spacing: 3px;
        }
        
        .player-info {
            text-align: right;
        }
        
        .player-name {
            font-weight: 600;
            color: #fff;
        }
        
        .player-status {
            font-size: 0.8rem;
            color: #aaa;
        }
        
        .container {
            max-width: 500px;
            margin: 0 auto;
        }
        
        .card {
            background: rgba(255, 255, 255, 0.05);
            border-radius: 20px;
            padding: 25px;
            margin-bottom: 20px;
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .card-title {
            font-size: 1.3rem;
            margin-bottom: 20px;
            text-align: center;
            color: #e94560;
        }
        
        /* Lobby styles */
        .players-list {
            list-style: none;
        }
        
        .player-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 12px 15px;
            background: rgba(255, 255, 255, 0.05);
            border-radius: 12px;
            margin-bottom: 10px;
            transition: all 0.3s;
        }
        
        .player-item.host {
            border: 1px solid #e94560;
        }
        
        .player-item.eliminated {
            opacity: 0.5;
            text-decoration: line-through;
        }
        
        .player-item.current-turn {
            background: rgba(233, 69, 96, 0.2);
            border: 1px solid #e94560;
        }
        
        .player-item .host-badge {
            background: #e94560;
            color: #fff;
            font-size: 0.7rem;
            padding: 3px 8px;
            border-radius: 5px;
            margin-left: 10px;
        }
        
        /* Secret word display */
        .secret-word {
            text-align: center;
            padding: 30px;
            background: linear-gradient(135deg, rgba(233, 69, 96, 0.2), rgba(255, 107, 107, 0.1));
            border-radius: 20px;
            margin-bottom: 20px;
            border: 2px solid rgba(233, 69, 96, 0.3);
        }
        
        .secret-word-label {
            font-size: 0.9rem;
            color: #aaa;
            margin-bottom: 10px;
        }
        
        .secret-word-value {
            font-size: 2.5rem;
            font-weight: bold;
            background: linear-gradient(45deg, #e94560, #ff6b6b);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }
        
        .secret-word.mrwhite {
            background: linear-gradient(135deg, rgba(108, 117, 125, 0.2), rgba(173, 181, 189, 0.1));
            border-color: rgba(108, 117, 125, 0.3);
        }
        
        .secret-word.mrwhite .secret-word-value {
            background: linear-gradient(45deg, #6c757d, #adb5bd);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
            font-size: 1.5rem;
        }
        
        /* Role badge */
        .role-badge {
            display: inline-block;
            padding: 5px 15px;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
            margin-top: 10px;
        }
        
        .role-civilian {
            background: linear-gradient(45deg, #28a745, #20c997);
        }
        
        .role-undercover {
            background: linear-gradient(45deg, #dc3545, #e94560);
        }
        
        .role-mrwhite {
            background: linear-gradient(45deg, #6c757d, #adb5bd);
        }
        
        /* Buttons */
        .btn {
            width: 100%;
            padding: 15px;
            border: none;
            border-radius: 12px;
            font-size: 1.1rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
            margin-top: 10px;
        }
        
        .btn-primary {
            background: linear-gradient(45deg, #e94560, #ff6b6b);
            color: #fff;
            box-shadow: 0 5px 20px rgba(233, 69, 96, 0.4);
        }
        
        .btn-primary:hover:not(:disabled) {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(233, 69, 96, 0.5);
        }
        
        .btn-secondary {
            background: rgba(255, 255, 255, 0.1);
            color: #fff;
            border: 1px solid rgba(255, 255, 255, 0.2);
        }
        
        .btn-danger {
            background: linear-gradient(45deg, #dc3545, #c82333);
            color: #fff;
        }
        
        .btn:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        
        /* Phase indicator */
        .phase-indicator {
            text-align: center;
            padding: 15px;
            background: rgba(233, 69, 96, 0.1);
            border-radius: 15px;
            margin-bottom: 20px;
            border: 1px solid rgba(233, 69, 96, 0.2);
        }
        
        .phase-name {
            font-size: 1.2rem;
            font-weight: 600;
            color: #e94560;
        }
        
        .phase-description {
            font-size: 0.85rem;
            color: #aaa;
            margin-top: 5px;
        }
        
        /* Vote buttons */
        .vote-grid {
            display: grid;
            grid-template-columns: 1fr;
            gap: 10px;
        }
        
        .vote-btn {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px 20px;
            background: rgba(255, 255, 255, 0.05);
            border: 2px solid transparent;
            border-radius: 12px;
            cursor: pointer;
            transition: all 0.3s;
            color: #fff;
            font-size: 1rem;
        }
        
        .vote-btn:hover {
            background: rgba(233, 69, 96, 0.1);
            border-color: rgba(233, 69, 96, 0.3);
        }
        
        .vote-btn.selected {
            background: rgba(233, 69, 96, 0.2);
            border-color: #e94560;
        }
        
        .vote-btn.voted-for {
            border-color: #28a745;
        }
        
        .vote-count {
            background: #e94560;
            padding: 5px 12px;
            border-radius: 15px;
            font-size: 0.9rem;
        }
        
        /* Messages */
        .message {
            text-align: center;
            padding: 20px;
            border-radius: 15px;
            margin-bottom: 20px;
        }
        
        .message-waiting {
            background: rgba(255, 193, 7, 0.1);
            border: 1px solid rgba(255, 193, 7, 0.3);
            color: #ffc107;
        }
        
        .message-success {
            background: rgba(40, 167, 69, 0.1);
            border: 1px solid rgba(40, 167, 69, 0.3);
            color: #28a745;
        }
        
        .message-danger {
            background: rgba(220, 53, 69, 0.1);
            border: 1px solid rgba(220, 53, 69, 0.3);
            color: #dc3545;
        }
        
        /* Victory screen */
        .victory-screen {
            text-align: center;
            padding: 30px 20px;
        }
        
        .victory-screen.you-won {
            background: linear-gradient(135deg, rgba(40, 167, 69, 0.15), rgba(0, 100, 0, 0.1));
            border: 2px solid #28a745;
        }
        
        .victory-screen.you-lost {
            background: linear-gradient(135deg, rgba(220, 53, 69, 0.15), rgba(139, 0, 0, 0.1));
            border: 2px solid #dc3545;
        }
        
        .personal-result {
            margin-bottom: 25px;
            padding-bottom: 20px;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .personal-icon {
            font-size: 4rem;
            margin-bottom: 10px;
        }
        
        .personal-title {
            font-size: 2.5rem;
            font-weight: bold;
            margin-bottom: 5px;
        }
        
        .personal-title.win {
            color: #28a745;
            text-shadow: 0 0 20px rgba(40, 167, 69, 0.5);
        }
        
        .personal-title.lose {
            color: #dc3545;
            text-shadow: 0 0 20px rgba(220, 53, 69, 0.5);
        }
        
        .personal-subtitle {
            color: #aaa;
            font-size: 1.1rem;
        }
        
        .winner-announcement {
            margin: 20px 0;
        }
        
        .victory-icon {
            font-size: 3rem;
            margin-bottom: 10px;
        }
        
        .victory-title {
            font-size: 1.5rem;
            margin-bottom: 10px;
            background: linear-gradient(45deg, #e94560, #ff6b6b);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }
        
        .words-reveal {
            background: rgba(255, 255, 255, 0.05);
            padding: 15px;
            border-radius: 12px;
            margin: 20px 0;
            text-align: left;
        }
        
        .words-reveal p {
            margin: 8px 0;
            color: #ddd;
        }
        
        .victory-subtitle {
            color: #aaa;
            margin-bottom: 30px;
        }
        
        .results-list {
            text-align: left;
            list-style: none;
        }
        
        .result-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 12px;
            background: rgba(255, 255, 255, 0.05);
            border-radius: 10px;
            margin-bottom: 8px;
        }
        
        .result-item.is-me {
            background: rgba(233, 69, 96, 0.2);
            border: 1px solid rgba(233, 69, 96, 0.4);
        }
        
        .result-role {
            font-size: 0.75rem;
            padding: 3px 10px;
            border-radius: 10px;
        }
        
        /* Mr White guess form */
        .guess-form {
            margin-top: 20px;
        }
        
        .guess-input {
            width: 100%;
            padding: 15px;
            border: 2px solid rgba(255, 255, 255, 0.1);
            border-radius: 12px;
            background: rgba(255, 255, 255, 0.05);
            color: #fff;
            font-size: 1.2rem;
            text-align: center;
            margin-bottom: 10px;
        }
        
        .guess-input:focus {
            outline: none;
            border-color: #e94560;
        }
        
        /* Loading spinner */
        .spinner {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid rgba(255, 255, 255, 0.3);
            border-radius: 50%;
            border-top-color: #fff;
            animation: spin 1s linear infinite;
            margin-right: 10px;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        /* Footer actions */
        .footer-actions {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            padding: 15px;
            background: linear-gradient(to top, rgba(26, 26, 46, 0.98), rgba(26, 26, 46, 0.8));
            backdrop-filter: blur(10px);
        }
        
        .footer-actions .container {
            display: flex;
            gap: 10px;
        }
        
        .footer-actions .btn {
            flex: 1;
            margin: 0;
        }
        
        /* Hide elements */
        .hidden {
            display: none !important;
        }
        
        /* Timer */
        .timer {
            font-size: 2rem;
            text-align: center;
            color: #e94560;
            font-weight: bold;
            margin: 20px 0;
        }
        
        /* Instructions */
        .instructions {
            background: rgba(255, 255, 255, 0.03);
            padding: 15px;
            border-radius: 12px;
            margin-top: 15px;
        }
        
        .instructions h4 {
            color: #e94560;
            margin-bottom: 10px;
            font-size: 0.9rem;
        }
        
        .instructions p {
            font-size: 0.85rem;
            color: #aaa;
            line-height: 1.5;
        }
        
        /* Eliminated player reveal */
        .eliminated-reveal {
            text-align: center;
            padding: 20px;
            background: rgba(220, 53, 69, 0.1);
            border: 1px solid rgba(220, 53, 69, 0.3);
            border-radius: 15px;
            margin-bottom: 20px;
        }
        
        .eliminated-name {
            font-size: 1.5rem;
            font-weight: bold;
            color: #dc3545;
        }
        
        .eliminated-role {
            margin-top: 10px;
        }
        
        /* Nouveaux écrans d'élimination personnalisés */
        .elimination-screen {
            text-align: center;
            padding: 40px 20px;
            border-radius: 20px;
            margin-bottom: 20px;
        }
        
        .elimination-screen.eliminated-you {
            background: linear-gradient(135deg, rgba(220, 53, 69, 0.2), rgba(139, 0, 0, 0.3));
            border: 2px solid #dc3545;
            animation: pulse-red 2s infinite;
        }
        
        .elimination-screen.good-elimination {
            background: linear-gradient(135deg, rgba(40, 167, 69, 0.2), rgba(0, 100, 0, 0.2));
            border: 2px solid #28a745;
        }
        
        .elimination-screen.bad-elimination {
            background: linear-gradient(135deg, rgba(255, 193, 7, 0.2), rgba(200, 150, 0, 0.2));
            border: 2px solid #ffc107;
        }
        
        .elimination-icon {
            font-size: 4rem;
            margin-bottom: 15px;
        }
        
        .elimination-title {
            font-size: 1.8rem;
            margin-bottom: 10px;
            color: #fff;
        }
        
        .elimination-subtitle {
            font-size: 1.1rem;
            color: #ccc;
            margin-bottom: 20px;
        }
        
        .elimination-role {
            margin: 20px 0;
        }
        
        .elimination-role .role-badge {
            font-size: 1rem;
            padding: 10px 25px;
        }
        
        @keyframes pulse-red {
            0%, 100% { box-shadow: 0 0 20px rgba(220, 53, 69, 0.3); }
            50% { box-shadow: 0 0 40px rgba(220, 53, 69, 0.6); }
        }
    </style>
</head>
<body>
    <div class="header">
        <div class="game-code" title="Code de la partie"><?= htmlspecialchars($gameCode) ?></div>
        <div class="player-info">
            <div class="player-name"><?= htmlspecialchars($player['name']) ?></div>
            <div class="player-status" id="player-status">
                <?= $isHost ? '👑 Hôte' : '👤 Joueur' ?>
            </div>
        </div>
    </div>
    
    <div class="container">
        <div id="game-content">
            <!-- Le contenu sera chargé dynamiquement -->
            <div class="card">
                <div style="text-align: center; padding: 20px;">
                    <div class="spinner"></div>
                    <p>Chargement...</p>
                </div>
            </div>
        </div>
    </div>
    
    <div class="footer-actions hidden" id="footer-actions">
        <div class="container">
            <!-- Les boutons d'action seront ajoutés ici -->
        </div>
    </div>
    
    <script>
        const playerId = '<?= $playerId ?>';
        const gameCode = '<?= $gameCode ?>';
        const isHost = <?= $isHost ? 'true' : 'false' ?>;
        let currentPhase = '';
        let refreshInterval;
        
        // Rafraîchir l'état du jeu
        async function refreshGame() {
            try {
                const response = await fetch('api.php?action=status&code=' + gameCode + '&player=' + playerId);
                const data = await response.json();
                
                if (data.error) {
                    if (data.error === 'game_not_found' || data.error === 'player_not_found') {
                        window.location.href = 'index.php';
                        return;
                    }
                    console.error(data.error);
                    return;
                }
                
                updateGameDisplay(data);
            } catch (error) {
                console.error('Erreur de connexion:', error);
            }
        }
        
        // Mettre à jour l'affichage
        function updateGameDisplay(data) {
            const content = document.getElementById('game-content');
            const footer = document.getElementById('footer-actions');
            
            // Sauvegarder la phase actuelle pour détecter les changements
            const previousPhase = currentPhase;
            currentPhase = data.phase;
            
            let html = '';
            let footerHtml = '';
            
            switch (data.phase) {
                case 'lobby':
                    html = renderLobby(data);
                    if (isHost) {
                        footerHtml = `<button class="btn btn-primary" onclick="startGame()" ${data.playerCount < 3 ? 'disabled' : ''}>
                            Lancer la partie (${data.playerCount} joueurs)
                        </button>`;
                    }
                    break;
                    
                case 'description':
                    html = renderDescriptionPhase(data);
                    if (data.isCurrentDescriber && !data.player.hasDescribed) {
                        footerHtml = `<button class="btn btn-primary" onclick="markDescribed()">
                            J'ai décrit mon mot ✓
                        </button>`;
                    }
                    if (isHost) {
                        footerHtml += `<button class="btn btn-secondary" onclick="changeWords()" style="margin-top: 10px;">
                            🔄 Changer les mots (nuls)
                        </button>`;
                    }
                    break;
                    
                case 'discussion':
                    html = renderDiscussionPhase(data);
                    if (isHost) {
                        footerHtml = `<button class="btn btn-primary" onclick="startVoting()">
                            Passer au vote
                        </button>
                        <button class="btn btn-secondary" onclick="changeWords()">
                            🔄 Changer les mots (nuls)
                        </button>`;
                    }
                    break;
                    
                case 'voting':
                    html = renderVotingPhase(data);
                    if (isHost && data.allVoted) {
                        footerHtml = `<button class="btn btn-primary" onclick="endVoting()">
                            Révéler les résultats
                        </button>`;
                    }
                    break;
                    
                case 'mrwhite_guess':
                    html = renderMrWhiteGuess(data);
                    break;
                    
                case 'reveal':
                    html = renderRevealPhase(data);
                    if (isHost) {
                        footerHtml = `<button class="btn btn-primary" onclick="nextRound()">
                            Tour suivant
                        </button>`;
                    }
                    break;
                    
                case 'ended':
                    html = renderEndGame(data);
                    footerHtml = `<button class="btn btn-primary" onclick="leaveGame()">
                        Quitter
                    </button>`;
                    if (isHost) {
                        footerHtml = `<button class="btn btn-secondary" onclick="restartGame()">Rejouer</button>` + footerHtml;
                    }
                    break;
            }
            
            content.innerHTML = html;
            
            if (footerHtml) {
                footer.innerHTML = '<div class="container">' + footerHtml + '</div>';
                footer.classList.remove('hidden');
            } else {
                footer.classList.add('hidden');
            }
        }
        
        // Render: Lobby
        function renderLobby(data) {
            let playersHtml = '';
            for (const p of data.players) {
                playersHtml += `
                    <li class="player-item ${p.isHost ? 'host' : ''}">
                        <span>
                            ${p.name}
                            ${p.isHost ? '<span class="host-badge">Hôte</span>' : ''}
                        </span>
                        <span>${p.id === playerId ? '(vous)' : ''}</span>
                    </li>
                `;
            }
            
            return `
                <div class="card">
                    <h2 class="card-title">👥 Joueurs en attente</h2>
                    <ul class="players-list">${playersHtml}</ul>
                    
                    <div class="instructions">
                        <h4>📱 Partagez le code</h4>
                        <p>Donnez le code <strong>${gameCode}</strong> aux autres joueurs pour qu'ils puissent rejoindre la partie.</p>
                    </div>
                </div>
                
                <div class="card">
                    <h2 class="card-title">📊 Distribution des rôles</h2>
                    <p style="text-align: center; color: #aaa;">
                        ${data.roleDistribution.civilians} Civils, 
                        ${data.roleDistribution.undercovers} Undercover(s), 
                        ${data.roleDistribution.mrwhites} Mr. White
                    </p>
                </div>
            `;
        }
        
        // Render: Phase de description
        function renderDescriptionPhase(data) {
            let secretWordHtml = '';
            if (data.player.role === 'mrwhite') {
                secretWordHtml = `
                    <div class="secret-word mrwhite">
                        <div class="secret-word-label">Votre rôle</div>
                        <div class="secret-word-value">👻 Mr. White</div>
                        <div class="role-badge role-mrwhite">Vous n'avez pas de mot - improvisez !</div>
                    </div>
                `;
            } else {
                secretWordHtml = `
                    <div class="secret-word">
                        <div class="secret-word-label">Votre mot secret</div>
                        <div class="secret-word-value">${data.player.word}</div>
                        <div class="role-badge role-${data.player.role}">${data.player.role === 'civilian' ? '👥 Civil' : '🕵️ Undercover'}</div>
                    </div>
                `;
            }
            
            let playersHtml = '';
            for (let i = 0; i < data.descriptionOrder.length; i++) {
                const p = data.descriptionOrder[i];
                const isCurrent = i === data.currentDescriber;
                const hasDescribed = p.hasDescribed;
                
                playersHtml += `
                    <li class="player-item ${isCurrent ? 'current-turn' : ''} ${!p.alive ? 'eliminated' : ''}">
                        <span>${p.name} ${p.id === playerId ? '(vous)' : ''}</span>
                        <span>
                            ${hasDescribed ? '✓' : (isCurrent ? '🎤' : '⏳')}
                        </span>
                    </li>
                `;
            }
            
            return `
                ${secretWordHtml}
                
                <div class="phase-indicator">
                    <div class="phase-name">🎤 Phase de description</div>
                    <div class="phase-description">Décrivez votre mot avec un mot ou une courte phrase</div>
                </div>
                
                <div class="card">
                    <h2 class="card-title">Ordre de passage</h2>
                    <ul class="players-list">${playersHtml}</ul>
                </div>
                
                ${data.isCurrentDescriber && !data.player.hasDescribed ? `
                    <div class="message message-success">
                        <strong>C'est votre tour !</strong><br>
                        Décrivez votre mot à voix haute, puis appuyez sur le bouton.
                    </div>
                ` : ''}
            `;
        }
        
        // Render: Phase de discussion
        function renderDiscussionPhase(data) {
            let secretWordHtml = '';
            if (data.player.role === 'mrwhite') {
                secretWordHtml = `
                    <div class="secret-word mrwhite">
                        <div class="secret-word-label">Votre rôle</div>
                        <div class="secret-word-value">👻 Mr. White</div>
                    </div>
                `;
            } else {
                secretWordHtml = `
                    <div class="secret-word">
                        <div class="secret-word-label">Votre mot secret</div>
                        <div class="secret-word-value">${data.player.word}</div>
                    </div>
                `;
            }
            
            return `
                ${secretWordHtml}
                
                <div class="phase-indicator">
                    <div class="phase-name">💬 Phase de discussion</div>
                    <div class="phase-description">Discutez et débattez pour trouver les infiltrés !</div>
                </div>
                
                <div class="card">
                    <h2 class="card-title">Conseils</h2>
                    <div class="instructions">
                        <p>🔍 Posez des questions pour identifier qui a un mot différent</p>
                        <p style="margin-top: 10px;">🤫 Ne révélez pas trop d'informations sur votre mot</p>
                        <p style="margin-top: 10px;">🎭 Les infiltrés doivent se fondre dans la masse</p>
                    </div>
                </div>
                
                ${isHost ? `
                    <div class="message message-waiting">
                        En tant qu'hôte, lancez le vote quand la discussion est terminée.
                    </div>
                ` : `
                    <div class="message message-waiting">
                        En attente que l'hôte lance le vote...
                    </div>
                `}
            `;
        }
        
        // Render: Phase de vote
        function renderVotingPhase(data) {
            let votingHtml = '';
            
            if (data.player.alive && !data.player.votedFor) {
                for (const p of data.players) {
                    if (p.alive && p.id !== playerId) {
                        votingHtml += `
                            <button class="vote-btn" onclick="vote('${p.id}')">
                                <span>${p.name}</span>
                                <span class="vote-count">${p.votes} vote(s)</span>
                            </button>
                        `;
                    }
                }
            }
            
            const hasVoted = data.player.votedFor;
            const skipped = data.player.votedFor === 'skip';
            
            return `
                <div class="phase-indicator">
                    <div class="phase-name">🗳️ Phase de vote</div>
                    <div class="phase-description">Votez pour éliminer un joueur suspect ou passez</div>
                </div>
                
                <div class="card">
                    <h2 class="card-title">Choisissez qui éliminer</h2>
                    
                    ${!data.player.alive ? `
                        <div class="message message-danger">
                            Vous êtes éliminé et ne pouvez pas voter.
                        </div>
                    ` : (hasVoted ? `
                        <div class="message message-success">
                            ${skipped ? '⏭️ Vous avez passé votre vote' : '✅ Vous avez voté !'}<br>
                            <small>En attente des autres joueurs...</small>
                        </div>
                    ` : '')}
                    
                    ${!hasVoted && data.player.alive ? `
                        <div class="vote-grid">
                            ${votingHtml}
                        </div>
                        
                        <button class="btn btn-secondary" onclick="vote('skip')" style="margin-top: 15px;">
                            ⏭️ Passer (ne pas voter)
                        </button>
                    ` : ''}
                </div>
                
                <div class="card">
                    <h2 class="card-title">📊 État du vote</h2>
                    <p style="text-align: center; color: #aaa;">
                        ${data.votedCount} / ${data.aliveCount} joueurs ont voté
                    </p>
                </div>
            `;
        }
        
        // Render: Mr. White devine le mot
        function renderMrWhiteGuess(data) {
            const isMrWhite = data.eliminatedPlayer && data.eliminatedPlayer.id === playerId && data.player.role === 'mrwhite';
            
            return `
                <div class="eliminated-reveal">
                    <div class="eliminated-name">👻 ${data.eliminatedPlayer.name}</div>
                    <div class="eliminated-role">
                        <span class="role-badge role-mrwhite">Mr. White</span>
                    </div>
                    <p style="margin-top: 15px; color: #aaa;">Mr. White a été découvert mais peut encore gagner en devinant le mot des Civils !</p>
                </div>
                
                ${isMrWhite ? `
                    <div class="card">
                        <h2 class="card-title">🎯 Devinez le mot des Civils !</h2>
                        <div class="guess-form">
                            <input type="text" id="mrwhite-guess" class="guess-input" 
                                   placeholder="Entrez le mot..." autocomplete="off">
                            <button class="btn btn-primary" onclick="submitGuess()">
                                Valider ma réponse
                            </button>
                        </div>
                    </div>
                ` : `
                    <div class="message message-waiting">
                        En attente de la réponse de Mr. White...
                    </div>
                `}
            `;
        }
        
        // Render: Révélation
        function renderRevealPhase(data) {
            let resultHtml = '';
            
            // Vérifier si le joueur actuel est l'éliminé
            const isEliminated = data.eliminatedPlayer && data.eliminatedPlayer.id === playerId;
            
            if (data.mrWhiteGuess) {
                resultHtml = `
                    <div class="card">
                        <h2 class="card-title">🎯 Tentative de Mr. White</h2>
                        <p style="text-align: center; font-size: 1.5rem; margin-bottom: 10px;">
                            "${data.mrWhiteGuess.guess}"
                        </p>
                        <p style="text-align: center; color: ${data.mrWhiteGuess.correct ? '#28a745' : '#dc3545'};">
                            ${data.mrWhiteGuess.correct ? '✅ Correct ! Mr. White gagne !' : '❌ Incorrect !'}
                        </p>
                    </div>
                `;
            }
            
            if (data.eliminatedPlayer) {
                if (isEliminated) {
                    // Écran pour le joueur éliminé
                    resultHtml += `
                        <div class="elimination-screen eliminated-you">
                            <div class="elimination-icon">💀</div>
                            <h2 class="elimination-title">Tu as été démasqué !</h2>
                            <p class="elimination-subtitle">Les autres joueurs t'ont éliminé...</p>
                            <div class="elimination-role">
                                <span class="role-badge role-${data.player.role}">
                                    ${data.player.role === 'civilian' ? '👥 Tu étais Civil' : 
                                      (data.player.role === 'undercover' ? '🕵️ Tu étais Undercover' : '👻 Tu étais Mr. White')}
                                </span>
                            </div>
                            ${data.player.role === 'civilian' ? `
                                <p style="color: #ff6b6b; margin-top: 15px;">
                                    ⚠️ Oups ! Les autres ont fait une erreur en t'éliminant !
                                </p>
                            ` : `
                                <p style="color: #aaa; margin-top: 15px;">
                                    Bien joué aux autres, ils t'ont trouvé !
                                </p>
                            `}
                        </div>
                    `;
                } else {
                    // Écran pour les autres joueurs
                    const eliminatedRole = data.eliminatedPlayer.role;
                    const wasInfiltrator = eliminatedRole === 'undercover' || eliminatedRole === 'mrwhite';
                    
                    resultHtml += `
                        <div class="elimination-screen ${wasInfiltrator ? 'good-elimination' : 'bad-elimination'}">
                            <div class="elimination-icon">${wasInfiltrator ? '🎉' : '😬'}</div>
                            <h2 class="elimination-title">
                                ${wasInfiltrator ? 'Bien joué !' : 'Oups...'}
                            </h2>
                            <p class="elimination-subtitle">
                                <strong>${data.eliminatedPlayer.name}</strong> a été éliminé
                            </p>
                            <div class="elimination-role">
                                <span class="role-badge role-${eliminatedRole}">
                                    ${eliminatedRole === 'civilian' ? '👥 C\'était un Civil !' : 
                                      (eliminatedRole === 'undercover' ? '🕵️ C\'était l\'Undercover !' : '👻 C\'était Mr. White !')}
                                </span>
                            </div>
                            ${wasInfiltrator ? `
                                <p style="color: #28a745; margin-top: 15px;">
                                    ✅ Vous avez bien identifié un infiltré !
                                </p>
                            ` : `
                                <p style="color: #ff6b6b; margin-top: 15px;">
                                    ❌ Vous avez éliminé un innocent...
                                </p>
                            `}
                        </div>
                    `;
                }
            } else {
                resultHtml += `
                    <div class="message message-waiting">
                        Aucune élimination ce tour (égalité ou pas de votes)
                    </div>
                `;
            }
            
            return `
                ${resultHtml}
                
                ${!data.winner ? `
                    <div class="message message-waiting" style="margin-top: 20px;">
                        ${isHost ? "Appuyez pour continuer au prochain tour" : "En attente de l'hôte..."}
                    </div>
                ` : ''}
            `;
        }
        
        // Render: Fin de partie
        function renderEndGame(data) {
            // Déterminer si le joueur actuel a gagné
            const myRole = data.player.role;
            let iWon = false;
            
            if (data.winner === 'civilians' && myRole === 'civilian') {
                iWon = true;
            } else if (data.winner === 'infiltrators' && (myRole === 'undercover' || myRole === 'mrwhite')) {
                iWon = true;
            } else if (data.winner === 'mrwhite' && myRole === 'mrwhite') {
                iWon = true;
            }
            
            let winnersText = '';
            let icon = '';
            
            switch (data.winner) {
                case 'civilians':
                    winnersText = 'Les Civils ont gagné !';
                    icon = '👥';
                    break;
                case 'infiltrators':
                    winnersText = 'Les Infiltrés ont gagné !';
                    icon = '🕵️';
                    break;
                case 'mrwhite':
                    winnersText = 'Mr. White a gagné !';
                    icon = '👻';
                    break;
            }
            
            let resultsHtml = '';
            for (const p of data.players) {
                let roleBadge = '';
                switch (p.role) {
                    case 'civilian':
                        roleBadge = '<span class="result-role role-civilian">👥 Civil</span>';
                        break;
                    case 'undercover':
                        roleBadge = '<span class="result-role role-undercover">🕵️ Undercover</span>';
                        break;
                    case 'mrwhite':
                        roleBadge = '<span class="result-role role-mrwhite">👻 Mr. White</span>';
                        break;
                }
                
                const isMe = p.id === playerId;
                resultsHtml += `
                    <li class="result-item ${!p.alive ? 'eliminated' : ''} ${isMe ? 'is-me' : ''}">
                        <span>${p.name} ${p.alive ? '' : '💀'} ${isMe ? '(toi)' : ''}</span>
                        ${roleBadge}
                    </li>
                `;
            }
            
            return `
                <div class="card victory-screen ${iWon ? 'you-won' : 'you-lost'}">
                    <div class="personal-result">
                        ${iWon ? `
                            <div class="personal-icon">🏆</div>
                            <h2 class="personal-title win">VICTOIRE !</h2>
                            <p class="personal-subtitle">Félicitations, tu as gagné !</p>
                        ` : `
                            <div class="personal-icon">😢</div>
                            <h2 class="personal-title lose">DÉFAITE...</h2>
                            <p class="personal-subtitle">Pas de chance, tu as perdu</p>
                        `}
                    </div>
                    
                    <div class="winner-announcement">
                        <div class="victory-icon">${icon}</div>
                        <h3 class="victory-title">${winnersText}</h3>
                    </div>
                    
                    <div class="words-reveal">
                        <p>📝 <strong>Mot des Civils :</strong> ${data.civilianWord}</p>
                        <p>🕵️ <strong>Mot des Undercovers :</strong> ${data.undercoverWord}</p>
                    </div>
                    
                    <h4 style="margin: 20px 0 10px; color: #aaa;">Résultats</h4>
                    <ul class="results-list">${resultsHtml}</ul>
                </div>
            `;
        }
        
        // Actions API
        async function apiCall(action, params = {}) {
            try {
                const formData = new FormData();
                formData.append('action', action);
                formData.append('code', gameCode);
                formData.append('player', playerId);
                for (const [key, value] of Object.entries(params)) {
                    formData.append(key, value);
                }
                
                const response = await fetch('api.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await response.json();
                
                if (data.error) {
                    alert(data.error);
                    return false;
                }
                
                refreshGame();
                return true;
            } catch (error) {
                console.error('Erreur:', error);
                return false;
            }
        }
        
        function startGame() {
            apiCall('start');
        }
        
        function markDescribed() {
            apiCall('described');
        }
        
        function startVoting() {
            apiCall('start_voting');
        }
        
        function vote(targetId) {
            apiCall('vote', { target: targetId });
        }
        
        function endVoting() {
            apiCall('end_voting');
        }
        
        function submitGuess() {
            const guess = document.getElementById('mrwhite-guess').value.trim();
            if (guess) {
                apiCall('mrwhite_guess', { guess: guess });
            }
        }
        
        function nextRound() {
            apiCall('next_round');
        }
        
        function restartGame() {
            apiCall('restart');
        }
        
        function changeWords() {
            if (confirm('Changer les mots ? Tout le monde recevra de nouveaux mots et devra recommencer les descriptions.')) {
                apiCall('change_words');
            }
        }
        
        function leaveGame() {
            window.location.href = 'leave.php';
        }
        
        // Initialisation
        refreshGame();
        refreshInterval = setInterval(refreshGame, 2000);
        
        // Empêcher le zoom sur iOS
        document.addEventListener('gesturestart', function(e) {
            e.preventDefault();
        });
    </script>
</body>
</html>
