<?php
require_once __DIR__ . '/config.php';

if (!isset($_SESSION['player_id']) || !isset($_SESSION['game_code'])) {
    header('Location: index.php');
    exit;
}

$playerId = $_SESSION['player_id'];
$gameCode = $_SESSION['game_code'];

$game = loadGame($gameCode);

if (!$game || !isset($game['players'][$playerId])) {
    unset($_SESSION['player_id'], $_SESSION['game_code'], $_SESSION['player_name']);
    header('Location: index.php');
    exit;
}

$player = $game['players'][$playerId];
$isHost = ($game['host'] === $playerId);
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>⚡ Reaction Duel - <?= htmlspecialchars($gameCode) ?></title>
    <style>
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
            -webkit-tap-highlight-color: transparent;
        }
        
        html, body {
            height: 100%;
            overflow: hidden;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: #0f0f23;
            color: #fff;
            touch-action: manipulation;
        }
        
        .game-container {
            display: flex;
            flex-direction: column;
            height: 100vh;
            height: 100dvh;
        }
        
        /* Header */
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px 20px;
            background: rgba(255, 255, 255, 0.03);
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .game-code {
            background: linear-gradient(45deg, #00f5a0, #00d9f5);
            color: #000;
            padding: 5px 12px;
            border-radius: 8px;
            font-weight: bold;
            font-size: 0.9rem;
            letter-spacing: 2px;
        }
        
        .round-info {
            font-size: 0.9rem;
            color: #888;
        }
        
        /* Scoreboard */
        .scoreboard {
            display: flex;
            justify-content: space-around;
            padding: 20px;
            background: rgba(255, 255, 255, 0.02);
        }
        
        .player-score {
            text-align: center;
            flex: 1;
        }
        
        .player-score.me {
            border-right: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .player-name {
            font-size: 1rem;
            color: #aaa;
            margin-bottom: 5px;
        }
        
        .player-name.me {
            color: #00f5a0;
        }
        
        .score {
            font-size: 3rem;
            font-weight: bold;
        }
        
        .score.me {
            color: #00f5a0;
        }
        
        .score.opponent {
            color: #f55;
        }
        
        .score-target {
            font-size: 0.8rem;
            color: #555;
            margin-top: 5px;
        }
        
        /* Zone de tap principale */
        .tap-zone {
            flex: 1;
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            padding: 20px;
            transition: background 0.1s;
            cursor: pointer;
            user-select: none;
        }
        
        .tap-zone.waiting {
            background: linear-gradient(180deg, #1a1a3e 0%, #0f0f23 100%);
        }
        
        .tap-zone.countdown {
            background: linear-gradient(180deg, #2a2a1e 0%, #1a1a0e 100%);
        }
        
        .tap-zone.red {
            background: linear-gradient(180deg, #4a1a1a 0%, #2a0a0a 100%);
        }
        
        .tap-zone.green {
            background: linear-gradient(180deg, #1a4a2a 0%, #0a2a1a 100%);
            animation: pulse-green 0.5s infinite;
        }
        
        .tap-zone.result {
            background: linear-gradient(180deg, #1a1a3e 0%, #0f0f23 100%);
        }
        
        @keyframes pulse-green {
            0%, 100% { background: linear-gradient(180deg, #1a4a2a 0%, #0a2a1a 100%); }
            50% { background: linear-gradient(180deg, #2a6a3a 0%, #1a4a2a 100%); }
        }
        
        .tap-content {
            text-align: center;
        }
        
        .tap-icon {
            font-size: 5rem;
            margin-bottom: 20px;
        }
        
        .tap-text {
            font-size: 1.8rem;
            font-weight: bold;
            margin-bottom: 10px;
        }
        
        .tap-subtext {
            font-size: 1rem;
            color: #888;
        }
        
        /* Countdown */
        .countdown {
            font-size: 8rem;
            font-weight: bold;
            color: #ffd700;
            text-shadow: 0 0 30px rgba(255, 215, 0, 0.5);
        }
        
        /* Result */
        .result-icon {
            font-size: 6rem;
            margin-bottom: 20px;
        }
        
        .result-text {
            font-size: 1.5rem;
            margin-bottom: 10px;
        }
        
        .result-detail {
            font-size: 1rem;
            color: #aaa;
            margin-bottom: 20px;
        }
        
        .reaction-time {
            font-size: 2.5rem;
            font-weight: bold;
            color: #00f5a0;
            margin: 15px 0;
        }
        
        /* Buttons */
        .btn {
            padding: 15px 40px;
            border: none;
            border-radius: 12px;
            font-size: 1.1rem;
            font-weight: bold;
            cursor: pointer;
            transition: all 0.3s;
            margin: 5px;
        }
        
        .btn-primary {
            background: linear-gradient(45deg, #00f5a0, #00d9f5);
            color: #000;
        }
        
        .btn-secondary {
            background: rgba(255, 255, 255, 0.1);
            color: #fff;
            border: 1px solid rgba(255, 255, 255, 0.2);
        }
        
        .btn:active {
            transform: scale(0.95);
        }
        
        /* Winner screen */
        .winner-screen {
            text-align: center;
        }
        
        .winner-trophy {
            font-size: 6rem;
            margin-bottom: 20px;
            animation: bounce 0.5s infinite alternate;
        }
        
        @keyframes bounce {
            from { transform: translateY(0); }
            to { transform: translateY(-20px); }
        }
        
        .winner-title {
            font-size: 2rem;
            margin-bottom: 10px;
        }
        
        .winner-title.win {
            color: #00f5a0;
        }
        
        .winner-title.lose {
            color: #f55;
        }
        
        .final-score {
            font-size: 3rem;
            margin: 20px 0;
        }
        
        /* Waiting for opponent */
        .waiting-opponent {
            text-align: center;
        }
        
        .waiting-spinner {
            width: 60px;
            height: 60px;
            border: 4px solid rgba(0, 245, 160, 0.2);
            border-top-color: #00f5a0;
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin: 0 auto 20px;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        .share-code {
            background: rgba(255, 255, 255, 0.05);
            padding: 20px;
            border-radius: 15px;
            margin-top: 20px;
        }
        
        .share-code p {
            color: #888;
            margin-bottom: 10px;
        }
        
        .big-code {
            font-size: 3rem;
            font-weight: bold;
            letter-spacing: 10px;
            color: #00f5a0;
        }
        
        /* Ready button */
        .ready-section {
            padding: 20px;
            text-align: center;
        }
        
        .ready-status {
            display: flex;
            justify-content: center;
            gap: 30px;
            margin-bottom: 20px;
        }
        
        .ready-player {
            text-align: center;
        }
        
        .ready-indicator {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            margin: 0 auto 10px;
        }
        
        .ready-indicator.waiting {
            background: rgba(255, 255, 255, 0.1);
            border: 2px dashed #555;
        }
        
        .ready-indicator.ready {
            background: linear-gradient(45deg, #00f5a0, #00d9f5);
        }
        
        /* False start */
        .false-start {
            color: #f55;
            animation: shake 0.5s;
        }
        
        @keyframes shake {
            0%, 100% { transform: translateX(0); }
            25% { transform: translateX(-10px); }
            75% { transform: translateX(10px); }
        }

        /* Leave button */
        .leave-btn {
            position: absolute;
            top: 15px;
            right: 15px;
            background: none;
            border: none;
            color: #555;
            font-size: 1.5rem;
            cursor: pointer;
            padding: 5px;
        }
    </style>
</head>
<body>
    <div class="game-container" id="game">
        <a href="leave.php" class="leave-btn">✕</a>
        
        <div class="header">
            <div class="game-code"><?= htmlspecialchars($gameCode) ?></div>
            <div class="round-info" id="round-info">Round 0</div>
        </div>
        
        <div class="scoreboard" id="scoreboard">
            <!-- Rempli par JS -->
        </div>
        
        <div class="tap-zone waiting" id="tap-zone">
            <div class="tap-content" id="tap-content">
                <!-- Rempli par JS -->
            </div>
        </div>
    </div>
    
    <script>
        const playerId = '<?= $playerId ?>';
        const gameCode = '<?= $gameCode ?>';
        
        let gameState = null;
        let canTap = false;
        let signalTimeout = null;
        let countdownInterval = null;
        let localCountdown = 3;
        
        const tapZone = document.getElementById('tap-zone');
        const tapContent = document.getElementById('tap-content');
        const scoreboard = document.getElementById('scoreboard');
        const roundInfo = document.getElementById('round-info');
        
        // Récupérer l'état du jeu
        async function fetchStatus() {
            try {
                const res = await fetch(`api.php?action=status&code=${gameCode}&player=${playerId}`);
                const data = await res.json();
                
                if (data.error) {
                    if (data.error === 'game_not_found' || data.error === 'player_not_found') {
                        window.location.href = 'index.php';
                    }
                    return;
                }
                
                gameState = data;
                render();
            } catch (e) {
                console.error(e);
            }
        }
        
        // API call
        async function apiCall(action, params = {}) {
            const formData = new FormData();
            formData.append('action', action);
            formData.append('code', gameCode);
            formData.append('player', playerId);
            for (const [k, v] of Object.entries(params)) {
                formData.append(k, v);
            }
            
            const res = await fetch('api.php', { method: 'POST', body: formData });
            return await res.json();
        }
        
        // Render
        function render() {
            if (!gameState) return;
            
            renderScoreboard();
            roundInfo.textContent = `Round ${gameState.round}`;
            
            switch (gameState.status) {
                case 'waiting':
                    if (gameState.playerCount < 2) {
                        renderWaitingOpponent();
                    } else {
                        renderReadyScreen();
                    }
                    break;
                case 'countdown':
                    renderCountdown();
                    break;
                case 'playing':
                    // Géré localement après countdown
                    break;
                case 'result':
                    renderResult();
                    break;
                case 'finished':
                    renderGameOver();
                    break;
            }
        }
        
        function renderScoreboard() {
            if (!gameState) return;
            
            const me = gameState.player;
            const opp = gameState.opponent;
            
            scoreboard.innerHTML = `
                <div class="player-score me">
                    <div class="player-name me">${me.name} (toi)</div>
                    <div class="score me">${me.score}</div>
                    <div class="score-target">/ ${gameState.roundsToWin} pour gagner</div>
                </div>
                <div class="player-score">
                    <div class="player-name">${opp ? opp.name : '???'}</div>
                    <div class="score opponent">${opp ? opp.score : 0}</div>
                    <div class="score-target">/ ${gameState.roundsToWin} pour gagner</div>
                </div>
            `;
        }
        
        function renderWaitingOpponent() {
            tapZone.className = 'tap-zone waiting';
            tapContent.innerHTML = `
                <div class="waiting-opponent">
                    <div class="waiting-spinner"></div>
                    <div class="tap-text">En attente d'un adversaire...</div>
                    <div class="share-code">
                        <p>Partage ce code :</p>
                        <div class="big-code">${gameCode}</div>
                    </div>
                </div>
            `;
        }
        
        function renderReadyScreen() {
            tapZone.className = 'tap-zone waiting';
            
            const me = gameState.player;
            const opp = gameState.opponent;
            
            tapContent.innerHTML = `
                <div class="ready-section">
                    <div class="tap-icon">⚡</div>
                    <div class="tap-text">Prêt pour le Round ${gameState.round + 1} ?</div>
                    
                    <div class="ready-status">
                        <div class="ready-player">
                            <div class="ready-indicator ${me.ready ? 'ready' : 'waiting'}">
                                ${me.ready ? '✓' : '?'}
                            </div>
                            <div>${me.name}</div>
                        </div>
                        <div class="ready-player">
                            <div class="ready-indicator ${opp && opp.ready ? 'ready' : 'waiting'}">
                                ${opp && opp.ready ? '✓' : '?'}
                            </div>
                            <div>${opp ? opp.name : '???'}</div>
                        </div>
                    </div>
                    
                    ${!me.ready ? `
                        <button class="btn btn-primary" onclick="setReady()">Je suis prêt !</button>
                    ` : `
                        <div class="tap-subtext">En attente de ${opp ? opp.name : 'l\'adversaire'}...</div>
                    `}
                </div>
            `;
        }
        
        function renderCountdown() {
            tapZone.className = 'tap-zone countdown';
            
            // Démarrer le countdown local
            if (!countdownInterval) {
                localCountdown = 3;
                updateCountdownDisplay();
                
                countdownInterval = setInterval(() => {
                    localCountdown--;
                    
                    if (localCountdown <= 0) {
                        clearInterval(countdownInterval);
                        countdownInterval = null;
                        startRound();
                    } else {
                        updateCountdownDisplay();
                    }
                }, 1000);
            }
        }
        
        function updateCountdownDisplay() {
            tapContent.innerHTML = `
                <div class="countdown">${localCountdown}</div>
                <div class="tap-subtext">Prépare-toi...</div>
            `;
        }
        
        async function startRound() {
            // Appeler l'API pour démarrer le round
            const result = await apiCall('start_round');
            
            if (result.success) {
                // Afficher l'écran rouge (attente)
                tapZone.className = 'tap-zone red';
                canTap = true;
                
                tapContent.innerHTML = `
                    <div class="tap-icon">🔴</div>
                    <div class="tap-text">ATTENDS...</div>
                    <div class="tap-subtext">Tape quand ça devient vert !</div>
                `;
                
                // Programmer le signal vert
                signalTimeout = setTimeout(() => {
                    showGreen();
                }, result.signalDelay);
            }
        }
        
        function showGreen() {
            if (gameState.status !== 'playing') return;
            
            tapZone.className = 'tap-zone green';
            tapContent.innerHTML = `
                <div class="tap-icon">🟢</div>
                <div class="tap-text">TAP !!!</div>
            `;
        }
        
        async function handleTap() {
            if (!canTap) return;
            canTap = false;
            
            // Annuler le timeout du signal si on tape avant
            if (signalTimeout) {
                clearTimeout(signalTimeout);
                signalTimeout = null;
            }
            
            // Envoyer le tap
            const result = await apiCall('tap');
            
            if (result.roundFinished) {
                // Le round est terminé, fetch le nouvel état
                await fetchStatus();
            } else {
                // Attendre l'autre joueur
                tapContent.innerHTML = `
                    <div class="tap-icon">⏳</div>
                    <div class="tap-text">En attente...</div>
                    <div class="tap-subtext">L'adversaire n'a pas encore tapé</div>
                `;
            }
        }
        
        function renderResult() {
            canTap = false;
            tapZone.className = 'tap-zone result';
            
            const iWon = gameState.roundWinner === playerId;
            const isDraw = gameState.roundWinner === null;
            
            let icon, title;
            if (isDraw) {
                icon = '🤝';
                title = 'Égalité !';
            } else if (iWon) {
                icon = '🎉';
                title = 'Tu as gagné ce round !';
            } else {
                icon = '😤';
                title = 'Perdu ce round...';
            }
            
            tapContent.innerHTML = `
                <div class="result-icon">${icon}</div>
                <div class="result-text ${iWon ? 'win' : ''}">${title}</div>
                <div class="result-detail">${gameState.lastRoundResult || ''}</div>
                <button class="btn btn-primary" onclick="nextRound()">Round suivant</button>
            `;
        }
        
        function renderGameOver() {
            canTap = false;
            tapZone.className = 'tap-zone result';
            
            const iWon = gameState.gameWinner === playerId;
            const me = gameState.player;
            const opp = gameState.opponent;
            
            tapContent.innerHTML = `
                <div class="winner-screen">
                    <div class="winner-trophy">${iWon ? '🏆' : '😢'}</div>
                    <div class="winner-title ${iWon ? 'win' : 'lose'}">
                        ${iWon ? 'VICTOIRE !' : 'DÉFAITE...'}
                    </div>
                    <div class="final-score">
                        <span style="color: #00f5a0">${me.score}</span>
                        <span style="color: #555"> - </span>
                        <span style="color: #f55">${opp.score}</span>
                    </div>
                    <button class="btn btn-primary" onclick="rematch()">🔄 Revanche</button>
                    <button class="btn btn-secondary" onclick="window.location.href='leave.php'">Quitter</button>
                </div>
            `;
        }
        
        // Actions
        async function setReady() {
            await apiCall('ready');
            await fetchStatus();
        }
        
        async function nextRound() {
            await apiCall('next_round');
            await fetchStatus();
        }
        
        async function rematch() {
            await apiCall('rematch');
            await fetchStatus();
        }
        
        // Event listeners
        tapZone.addEventListener('click', () => {
            if (canTap) {
                handleTap();
            }
        });
        
        tapZone.addEventListener('touchstart', (e) => {
            if (canTap) {
                e.preventDefault();
                handleTap();
            }
        }, { passive: false });
        
        // Polling
        fetchStatus();
        setInterval(() => {
            if (gameState && (gameState.status === 'waiting' || gameState.status === 'countdown' || gameState.status === 'result')) {
                fetchStatus();
            }
        }, 500);
        
        // Polling plus lent pour les autres états
        setInterval(() => {
            if (gameState && gameState.status === 'playing') {
                fetchStatus();
            }
        }, 300);
    </script>
</body>
</html>
