<?php
/**
 * Configuration du jeu Reaction Duel
 */

// Activer les erreurs pour le debug
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Démarrer la session
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Répertoire pour stocker les parties
define('GAMES_DIR', __DIR__ . '/games/');

// Créer le répertoire s'il n'existe pas
if (!is_dir(GAMES_DIR)) {
    if (!mkdir(GAMES_DIR, 0777, true)) {
        die('Erreur: Impossible de créer le dossier games/');
    }
    chmod(GAMES_DIR, 0777);
}

// Vérifier que le dossier est writable
if (!is_writable(GAMES_DIR)) {
    chmod(GAMES_DIR, 0777);
}

// Configuration du jeu
define('ROUNDS_TO_WIN', 5);           // Nombre de rounds pour gagner
define('MIN_DELAY_MS', 2000);         // Délai minimum avant signal (ms)
define('MAX_DELAY_MS', 6000);         // Délai maximum avant signal (ms)
define('FALSE_START_PENALTY', 1);     // Points perdus pour faux départ

/**
 * Génère un code de partie aléatoire
 */
function generateGameCode(): string {
    $chars = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
    $code = '';
    for ($i = 0; $i < 4; $i++) {
        $code .= $chars[random_int(0, strlen($chars) - 1)];
    }
    return $code;
}

/**
 * Récupère le chemin du fichier d'une partie
 */
function getGameFilePath(string $code): string {
    return GAMES_DIR . strtoupper($code) . '.json';
}

/**
 * Charge une partie depuis le fichier JSON
 */
function loadGame(string $code): ?array {
    $path = getGameFilePath($code);
    if (!file_exists($path)) {
        return null;
    }
    $content = file_get_contents($path);
    return json_decode($content, true);
}

/**
 * Sauvegarde une partie dans le fichier JSON
 */
function saveGame(string $code, array $game): bool {
    $path = getGameFilePath($code);
    return file_put_contents($path, json_encode($game, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)) !== false;
}

/**
 * Supprime les parties anciennes (plus de 2h)
 */
function cleanOldGames(): void {
    $files = glob(GAMES_DIR . '*.json');
    $now = time();
    foreach ($files as $file) {
        if ($now - filemtime($file) > 7200) { // 2 heures
            unlink($file);
        }
    }
}

// Nettoyer les anciennes parties périodiquement
if (random_int(1, 10) === 1) {
    cleanOldGames();
}
