<?php
require_once __DIR__ . '/config.php';

header('Content-Type: application/json');

function jsonResponse($data) {
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

function jsonError($message) {
    jsonResponse(['error' => $message]);
}

$action = $_REQUEST['action'] ?? '';
$code = strtoupper($_REQUEST['code'] ?? '');
$playerId = $_REQUEST['player'] ?? '';

// Status - récupérer l'état du jeu
if ($action === 'status') {
    if (!$code || !$playerId) {
        jsonError('missing_params');
    }
    
    $game = loadGame($code);
    
    if (!$game) {
        jsonError('game_not_found');
    }
    
    if (!isset($game['players'][$playerId])) {
        jsonError('player_not_found');
    }
    
    $players = [];
    $opponent = null;
    
    foreach ($game['players'] as $pid => $p) {
        $players[] = [
            'id' => $pid,
            'name' => $p['name'],
            'score' => $p['score'],
            'ready' => $p['ready'],
            'tapped' => $p['tapped'],
            'falseStart' => $p['falseStart'],
            'isMe' => ($pid === $playerId)
        ];
        
        if ($pid !== $playerId) {
            $opponent = $p;
        }
    }
    
    $response = [
        'code' => $code,
        'status' => $game['status'],
        'round' => $game['round'],
        'players' => $players,
        'player' => $game['players'][$playerId],
        'opponent' => $opponent,
        'isHost' => ($game['host'] === $playerId),
        'playerCount' => count($game['players']),
        'roundWinner' => $game['roundWinner'],
        'gameWinner' => $game['gameWinner'],
        'lastRoundResult' => $game['lastRoundResult'],
        'signalDelay' => $game['signalDelay'],
        'roundStartTime' => $game['roundStartTime'],
        'roundsToWin' => ROUNDS_TO_WIN
    ];
    
    jsonResponse($response);
}

// Autres actions
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonError('method_not_allowed');
}

if (!$code || !$playerId) {
    jsonError('missing_params');
}

$game = loadGame($code);

if (!$game) {
    jsonError('Partie non trouvée');
}

if (!isset($game['players'][$playerId])) {
    jsonError('Joueur non trouvé');
}

switch ($action) {
    
    case 'ready':
        // Marquer le joueur comme prêt
        $game['players'][$playerId]['ready'] = true;
        
        // Vérifier si les 2 joueurs sont prêts
        $allReady = (count($game['players']) === 2);
        foreach ($game['players'] as $p) {
            if (!$p['ready']) {
                $allReady = false;
                break;
            }
        }
        
        if ($allReady) {
            $game['status'] = 'countdown';
        }
        
        $game['updatedAt'] = time();
        saveGame($code, $game);
        jsonResponse(['success' => true]);
        break;
        
    case 'start_round':
        // Démarrer un nouveau round (appelé après le countdown)
        if (count($game['players']) !== 2) {
            jsonError('Il faut 2 joueurs');
        }
        
        // Reset les états des joueurs
        foreach ($game['players'] as &$p) {
            $p['tapped'] = false;
            $p['tapTime'] = null;
            $p['falseStart'] = false;
        }
        unset($p);
        
        $game['round']++;
        $game['status'] = 'playing';
        $game['roundWinner'] = null;
        $game['lastRoundResult'] = null;
        
        // Générer le délai aléatoire (entre 2 et 6 secondes)
        $game['signalDelay'] = random_int(MIN_DELAY_MS, MAX_DELAY_MS);
        $game['roundStartTime'] = round(microtime(true) * 1000); // Timestamp en ms
        $game['signalTime'] = $game['roundStartTime'] + $game['signalDelay'];
        
        $game['updatedAt'] = time();
        saveGame($code, $game);
        jsonResponse(['success' => true, 'signalDelay' => $game['signalDelay']]);
        break;
        
    case 'tap':
        // Le joueur a tapé
        if ($game['status'] !== 'playing') {
            jsonError('Pas en jeu');
        }
        
        if ($game['players'][$playerId]['tapped']) {
            jsonError('Déjà tapé');
        }
        
        $now = round(microtime(true) * 1000);
        $game['players'][$playerId]['tapped'] = true;
        $game['players'][$playerId]['tapTime'] = $now;
        
        // Vérifier si c'est un faux départ
        if ($now < $game['signalTime']) {
            $game['players'][$playerId]['falseStart'] = true;
        }
        
        // Vérifier si le round est terminé (les 2 ont tapé ou faux départ)
        $player1 = null;
        $player2 = null;
        foreach ($game['players'] as $pid => $p) {
            if ($player1 === null) {
                $player1 = ['id' => $pid, 'data' => $p];
            } else {
                $player2 = ['id' => $pid, 'data' => $p];
            }
        }
        
        $roundFinished = false;
        $winnerId = null;
        $resultMessage = '';
        
        // Cas 1: Faux départ du joueur actuel
        if ($game['players'][$playerId]['falseStart']) {
            $roundFinished = true;
            // L'autre joueur gagne
            $winnerId = ($playerId === $player1['id']) ? $player2['id'] : $player1['id'];
            $resultMessage = $game['players'][$playerId]['name'] . ' a fait un faux départ ! 💥';
        }
        // Cas 2: Les deux ont tapé (sans faux départ du joueur actuel)
        elseif ($player1['data']['tapped'] && $player2['data']['tapped']) {
            $roundFinished = true;
            
            // Vérifier les faux départs
            if ($player1['data']['falseStart'] && !$player2['data']['falseStart']) {
                $winnerId = $player2['id'];
                $resultMessage = $player1['data']['name'] . ' a fait un faux départ ! 💥';
            } elseif ($player2['data']['falseStart'] && !$player1['data']['falseStart']) {
                $winnerId = $player1['id'];
                $resultMessage = $player2['data']['name'] . ' a fait un faux départ ! 💥';
            } elseif ($player1['data']['falseStart'] && $player2['data']['falseStart']) {
                // Les deux ont fait un faux départ, le moins pire gagne (ou égalité)
                $winnerId = null;
                $resultMessage = 'Double faux départ ! Égalité 🤯';
            } else {
                // Comparer les temps de réaction
                $time1 = $player1['data']['tapTime'] - $game['signalTime'];
                $time2 = $player2['data']['tapTime'] - $game['signalTime'];
                
                if ($time1 < $time2) {
                    $winnerId = $player1['id'];
                    $resultMessage = $player1['data']['name'] . ' gagne ! (' . $time1 . ' ms vs ' . $time2 . ' ms)';
                } elseif ($time2 < $time1) {
                    $winnerId = $player2['id'];
                    $resultMessage = $player2['data']['name'] . ' gagne ! (' . $time2 . ' ms vs ' . $time1 . ' ms)';
                } else {
                    $winnerId = null;
                    $resultMessage = 'Égalité parfaite ! ' . $time1 . ' ms 🤯';
                }
            }
        }
        
        if ($roundFinished) {
            $game['status'] = 'result';
            $game['roundWinner'] = $winnerId;
            $game['lastRoundResult'] = $resultMessage;
            
            // Ajouter le point au gagnant
            if ($winnerId) {
                $game['players'][$winnerId]['score']++;
                
                // Vérifier victoire finale
                if ($game['players'][$winnerId]['score'] >= ROUNDS_TO_WIN) {
                    $game['status'] = 'finished';
                    $game['gameWinner'] = $winnerId;
                }
            }
        }
        
        $game['updatedAt'] = time();
        saveGame($code, $game);
        
        $response = ['success' => true];
        if ($roundFinished) {
            $response['roundFinished'] = true;
            $response['result'] = $resultMessage;
        }
        jsonResponse($response);
        break;
        
    case 'next_round':
        // Préparer le prochain round
        if ($game['status'] !== 'result') {
            jsonError('Pas en phase résultat');
        }
        
        // Reset ready
        foreach ($game['players'] as &$p) {
            $p['ready'] = false;
            $p['tapped'] = false;
            $p['tapTime'] = null;
            $p['falseStart'] = false;
        }
        unset($p);
        
        $game['status'] = 'waiting';
        $game['roundWinner'] = null;
        $game['updatedAt'] = time();
        
        saveGame($code, $game);
        jsonResponse(['success' => true]);
        break;
        
    case 'rematch':
        // Recommencer une nouvelle partie
        foreach ($game['players'] as &$p) {
            $p['score'] = 0;
            $p['ready'] = false;
            $p['tapped'] = false;
            $p['tapTime'] = null;
            $p['falseStart'] = false;
        }
        unset($p);
        
        $game['status'] = 'waiting';
        $game['round'] = 0;
        $game['roundWinner'] = null;
        $game['gameWinner'] = null;
        $game['lastRoundResult'] = null;
        $game['updatedAt'] = time();
        
        saveGame($code, $game);
        jsonResponse(['success' => true]);
        break;
        
    default:
        jsonError('Action inconnue');
}

