<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Convertisseur JSON Bitwarden → Keeper</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Arial, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            justify-content: center;
            align-items: center;
            padding: 20px;
        }
        
        .container {
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            padding: 40px;
            max-width: 900px;
            width: 100%;
        }
        
        h1 {
            color: #333;
            margin-bottom: 10px;
            font-size: 28px;
            text-align: center;
        }
        
        .subtitle {
            color: #666;
            text-align: center;
            margin-bottom: 30px;
            font-size: 14px;
        }
        
        .info-box {
            background: #f0f4ff;
            border-left: 4px solid #667eea;
            padding: 15px;
            margin-bottom: 25px;
            border-radius: 5px;
        }
        
        .info-box h3 {
            color: #667eea;
            margin-bottom: 10px;
            font-size: 16px;
        }
        
        .example {
            background: #263238;
            color: #aed581;
            padding: 10px;
            border-radius: 5px;
            font-family: 'Courier New', monospace;
            font-size: 11px;
            margin: 8px 0;
            overflow-x: auto;
            white-space: pre;
        }
        
        .prefix-section {
            background: #fff8e1;
            border-left: 4px solid #ffa726;
            padding: 15px;
            margin: 20px 0;
            border-radius: 5px;
        }
        
        .prefix-section h3 {
            color: #f57c00;
            margin-bottom: 10px;
            font-size: 16px;
        }
        
        .prefix-input {
            width: 100%;
            padding: 12px;
            border: 2px solid #ffa726;
            border-radius: 8px;
            font-size: 14px;
            margin-top: 10px;
            transition: border-color 0.3s;
        }
        
        .prefix-input:focus {
            outline: none;
            border-color: #ff9800;
            box-shadow: 0 0 0 3px rgba(255, 152, 0, 0.1);
        }
        
        .prefix-examples {
            font-size: 12px;
            color: #666;
            margin-top: 8px;
            line-height: 1.5;
        }
        
        .prefix-examples code {
            background: #f5f5f5;
            padding: 2px 5px;
            border-radius: 3px;
            color: #d84315;
        }
        
        .upload-area {
            border: 2px dashed #667eea;
            border-radius: 10px;
            padding: 40px;
            text-align: center;
            transition: all 0.3s ease;
            cursor: pointer;
            background: #fafafa;
            margin: 20px 0;
        }
        
        .upload-area:hover {
            border-color: #764ba2;
            background: #f5f5ff;
        }
        
        .upload-area.dragover {
            background: #f0f0ff;
            border-color: #764ba2;
        }
        
        .upload-icon {
            font-size: 48px;
            color: #667eea;
            margin-bottom: 20px;
        }
        
        .file-input {
            display: none;
        }
        
        .btn {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            padding: 12px 30px;
            border-radius: 25px;
            font-size: 16px;
            cursor: pointer;
            transition: transform 0.2s ease;
            display: inline-block;
            text-decoration: none;
            margin: 5px;
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.2);
        }
        
        .btn:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        
        .convert-buttons {
            display: none;
            margin-top: 20px;
        }
        
        .convert-btn {
            width: 100%;
            margin-top: 10px;
            padding: 15px 30px;
            font-size: 17px;
            font-weight: 600;
        }
        
        .convert-all {
            background: linear-gradient(135deg, #4caf50 0%, #45a049 100%);
        }
        
        .convert-notes {
            background: linear-gradient(135deg, #ff9800 0%, #f57c00 100%);
        }
        
        .result {
            margin-top: 30px;
            padding: 20px;
            background: #f0fff4;
            border-radius: 10px;
            border: 1px solid #4caf50;
            display: none;
        }
        
        .result.show {
            display: block;
        }
        
        .error {
            background: #ffebee;
            border-color: #f44336;
        }
        
        .stats {
            display: grid;
            grid-template-columns: 1fr 1fr 1fr;
            gap: 10px;
            margin: 15px 0;
        }
        
        .stat {
            padding: 10px;
            background: #f5f5f5;
            border-radius: 5px;
            text-align: center;
        }
        
        .stat-value {
            font-size: 24px;
            font-weight: bold;
            color: #667eea;
        }
        
        .stat-label {
            font-size: 12px;
            color: #999;
            margin-top: 5px;
        }
        
        .preview {
            background: #263238;
            color: #aed581;
            padding: 15px;
            border-radius: 5px;
            font-family: 'Courier New', monospace;
            font-size: 11px;
            max-height: 400px;
            overflow-y: auto;
            margin-top: 20px;
            white-space: pre;
        }
        
        .field-mapping {
            background: #e8f5e9;
            border-left: 4px solid #4caf50;
            padding: 15px;
            margin: 15px 0;
            border-radius: 5px;
        }
        
        .field-mapping h4 {
            color: #2e7d32;
            margin-bottom: 10px;
        }
        
        .field-item {
            font-family: 'Courier New', monospace;
            font-size: 12px;
            margin: 5px 0;
            color: #1b5e20;
        }
        
        .file-loaded {
            background: #e8f5e9;
            color: #2e7d32;
            padding: 10px;
            border-radius: 5px;
            margin-top: 10px;
            text-align: center;
            font-weight: 600;
        }
        
        .warning-box {
            background: #fff3cd;
            border-left: 4px solid #ffc107;
            padding: 15px;
            margin: 15px 0;
            border-radius: 5px;
        }
        
        .warning-box h4 {
            color: #856404;
            margin-bottom: 10px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔄 Convertisseur JSON Bitwarden → Keeper</h1>
        <p class="subtitle">Conversion avec formats corrects pour Logins et Notes</p>
        
        <div class="info-box">
            <h3>📋 Formats de sortie garantis :</h3>
            <div class="example">// Format pour les LOGINS:
{
  "title": "Mon site",
  "$type": "login",                    // ✅ Avec $
  "login": "utilisateur",
  "password": "motdepasse",
  "login_url": "https://site.com",
  "notes": "Ligne 1\nLigne 2",         // ✅ \n préservés
  "custom_fields": {
    "Security Group": "Public",
    "Code": "123\n456\n789"
  },
  "folders": [{"folder": "Dossier"}]   // ✅ Tableau d'objets
}

// Format pour les NOTES:
{
  "title": "Mes infos",
  "type": "note",                      // ✅ Sans $ pour les notes
  "notes": "Info 1\nInfo 2\nInfo 3",   // ✅ \n préservés
  "folders": [{"folder": "Dossier"}]   // ✅ Tableau d'objets (identique aux logins)
}</div>
        </div>
        
        <!-- Section Préfixe de dossier -->
        <div class="prefix-section">
            <h3>📁 Préfixe de dossier (optionnel)</h3>
            <input type="text" 
                   id="folderPrefix" 
                   class="prefix-input"
                   placeholder="Ex: ITELIA - TECHNIQUE ou CLIENTS\ENTREPRISE_X\2024"
                   value="">
            <div class="prefix-examples">
                <strong>Exemples d'utilisation :</strong><br>
                • Laissez vide = Structure originale conservée<br>
                • <code>ITELIA - TECHNIQUE</code> → Tous les éléments dans ce dossier<br>
                • <code>CLIENTS\ENTREPRISE_X</code> → Structure avec sous-dossiers<br>
                • <code>TEST\IMPORT_2024</code> → Pour un import de test
            </div>
        </div>
        
        <!-- Zone d'upload -->
        <div class="upload-area" id="uploadArea">
            <div class="upload-icon">📄</div>
            <p>Glissez votre fichier JSON Bitwarden ici ou cliquez pour sélectionner</p>
            <input type="file" id="jsonFile" class="file-input" accept=".json">
            <button type="button" class="btn" onclick="document.getElementById('jsonFile').click()">
                Choisir un fichier JSON
            </button>
        </div>
        
        <div class="file-loaded" id="fileLoaded" style="display:none;">
            ✅ Fichier chargé - <span id="fileName"></span>
        </div>
        
        <!-- Boutons de conversion -->
        <div class="convert-buttons" id="convertButtons">
            <button type="button" 
                    class="btn convert-btn convert-all" 
                    onclick="convertToKeeper(false)">
                🚀 Convertir TOUT (Logins + Notes)
            </button>
            <button type="button" 
                    class="btn convert-btn convert-notes"
                    onclick="convertToKeeper(true)">
                📝 Convertir uniquement les NOTES
            </button>
        </div>
        
        <!-- Statistiques -->
        <div class="stats" id="stats" style="display:none;">
            <div class="stat">
                <div class="stat-value" id="itemCount">0</div>
                <div class="stat-label">Entrées</div>
            </div>
            <div class="stat">
                <div class="stat-value" id="folderCount">0</div>
                <div class="stat-label">Dossiers</div>
            </div>
            <div class="stat">
                <div class="stat-value" id="fieldCount">0</div>
                <div class="stat-label">Champs custom</div>
            </div>
        </div>
        
        <!-- Avertissement si des retours à la ligne sont détectés -->
        <div class="warning-box" id="newlineWarning" style="display:none;">
            <h4>⚠️ Retours à la ligne détectés</h4>
            <p style="font-size: 13px;">Les caractères <code>\n</code> ont été préservés dans les notes et champs personnalisés pour maintenir le formatage.</p>
        </div>
        
        <!-- Mapping des champs -->
        <div class="field-mapping" id="fieldMapping" style="display:none;">
            <h4>Champs personnalisés détectés :</h4>
            <div id="fieldList"></div>
        </div>
        
        <!-- Aperçu -->
        <div class="preview" id="preview" style="display:none;"></div>
        
        <!-- Résultat -->
        <div class="result" id="result">
            <h3 id="resultTitle">Conversion terminée</h3>
            <p id="resultMessage"></p>
            <div style="text-align: center; margin-top: 20px;">
                <a href="#" id="downloadJSON" class="btn" download>📥 Télécharger JSON</a>
                <a href="#" id="downloadCSV" class="btn" download>📥 Télécharger CSV</a>
            </div>
        </div>
    </div>

    <script>
        let bitwardenData = null;
        let keeperData = null;

        // Gestion du drag & drop
        document.getElementById('uploadArea').addEventListener('dragover', e => {
            e.preventDefault();
            e.currentTarget.classList.add('dragover');
        });

        document.getElementById('uploadArea').addEventListener('dragleave', e => {
            e.currentTarget.classList.remove('dragover');
        });

        document.getElementById('uploadArea').addEventListener('drop', e => {
            e.preventDefault();
            e.currentTarget.classList.remove('dragover');
            if (e.dataTransfer.files.length > 0) {
                handleFile(e.dataTransfer.files[0]);
            }
        });

        document.getElementById('jsonFile').addEventListener('change', e => {
            if (e.target.files.length > 0) {
                handleFile(e.target.files[0]);
            }
        });

        function handleFile(file) {
            const reader = new FileReader();
            reader.onload = e => {
                try {
                    bitwardenData = JSON.parse(e.target.result);
                    // Afficher le nom du fichier et les boutons de conversion
                    document.getElementById('fileName').textContent = file.name;
                    document.getElementById('fileLoaded').style.display = 'block';
                    document.getElementById('convertButtons').style.display = 'block';
                    // Réinitialiser les résultats précédents
                    document.getElementById('result').classList.remove('show');
                    document.getElementById('stats').style.display = 'none';
                    document.getElementById('fieldMapping').style.display = 'none';
                    document.getElementById('preview').style.display = 'none';
                    document.getElementById('newlineWarning').style.display = 'none';
                } catch (error) {
                    showError('Erreur lors de la lecture du fichier JSON: ' + error.message);
                }
            };
            reader.readAsText(file);
        }

        function processText(str, preserveNewlines = false) {
            if (!str) return '';
            
            // Convertir en string
            let result = String(str);
            
            // Nettoyer les caractères spéciaux MAIS garder \n, \r, \t
            result = result
                .replace(/[–—]/g, '-')  // Tirets spéciaux
                .replace(/[""]/g, '"')  // Guillemets courbes
                .replace(/['']/g, "'")  // Apostrophes courbes
                .replace(/[\u0000-\u0008\u000B\u000C\u000E-\u001F\u007F-\u009F]/g, ''); // Caractères de contrôle SAUF \t (09), \n (0A), \r (0D)
            
            return result;
        }

        function convertToKeeper(notesOnly = false) {
            if (!bitwardenData) {
                showError('Aucun fichier chargé');
                return;
            }
            
            const result = {
                records: []
            };
            
            // Récupérer le préfixe
            const prefix = document.getElementById('folderPrefix').value.trim();
            const cleanPrefix = prefix ? prefix.replace(/\//g, '\\') : '';
            
            // Créer le mapping des collections vers dossiers
            const collections = {};
            if (bitwardenData.collections) {
                bitwardenData.collections.forEach(c => {
                    collections[c.id] = c.name;
                });
            }
            
            // Créer le mapping des dossiers (pour les comptes personnels)
            const folders = {};
            if (bitwardenData.folders) {
                bitwardenData.folders.forEach(f => {
                    folders[f.id] = f.name;
                });
            }
            
            let itemCount = 0;
            let loginCount = 0;
            let noteCount = 0;
            let totalFieldCount = 0;
            let hasNewlines = false;
            const fieldExamples = [];
            
            // Convertir chaque item
            if (bitwardenData.items) {
                bitwardenData.items.forEach(item => {
                    // Si notesOnly, ignorer les logins (type 1)
                    if (notesOnly && item.type !== 2) return;
                    // Si pas notesOnly, prendre logins (type 1) et notes (type 2)
                    if (!notesOnly && item.type !== 1 && item.type !== 2) return;
                    
                    // Déterminer le dossier
                    let folderPath = '';
                    
                    // D'abord vérifier les collections (pour les organisations)
                    if (item.collectionIds && item.collectionIds.length > 0) {
                        const collectionNames = item.collectionIds
                            .map(id => collections[id])
                            .filter(name => name)
                            .join('\\');
                        if (collectionNames) {
                            folderPath = collectionNames;
                        }
                    }
                    
                    // Si pas de collection, utiliser le dossier
                    if (!folderPath && item.folderId && folders[item.folderId]) {
                        folderPath = folders[item.folderId];
                    }
                    
                    // Nettoyer le chemin du dossier
                    if (folderPath) {
                        folderPath = folderPath.replace(/[,\/]/g, '\\');
                    }
                    
                    // Appliquer le préfixe
                    if (cleanPrefix) {
                        if (folderPath) {
                            folderPath = cleanPrefix + '\\' + folderPath;
                        } else {
                            folderPath = cleanPrefix;
                        }
                    } else if (!folderPath) {
                        folderPath = 'Imported';
                    }
                    
                    // Créer l'enregistrement Keeper selon le type
                    const record = {
                        title: processText(item.name || 'Sans titre')
                    };
                    
                    if (item.type === 1) {
                        // Type LOGIN
                        record["$type"] = "login";  // Avec $ pour les logins
                        record.login = item.login?.username || '';
                        record.password = item.login?.password || '';
                        record.login_url = item.login?.uris?.[0]?.uri || '';
                        
                        // Préserver les \n dans les notes
                        if (item.notes) {
                            record.notes = processText(item.notes, true);
                            if (item.notes.includes('\n')) {
                                hasNewlines = true;
                            }
                        } else {
                            record.notes = '';
                        }
                        
                        // Créer les custom_fields
                        const customFields = {};
                        
                        // TOTP
                        if (item.login?.totp) {
                            const totp = item.login.totp;
                            if (totp.startsWith('otpauth://')) {
                                customFields['$oneTimeCode'] = totp;
                            } else {
                                const name = item.name?.replace(/[^a-zA-Z0-9]/g, '') || 'Account';
                                customFields['$oneTimeCode'] = `otpauth://totp/${name}?secret=${totp}&issuer=${name}&algorithm=SHA1&digits=6&period=30`;
                            }
                        }
                        
                        // Champs personnalisés Bitwarden
                        if (item.fields && Array.isArray(item.fields)) {
                            item.fields.forEach(field => {
                                if (field.name && field.value !== undefined && field.value !== null) {
                                    const fieldName = processText(field.name);
                                    const fieldValue = processText(String(field.value), true);
                                    customFields[fieldName] = fieldValue;
                                    
                                    if (String(field.value).includes('\n')) {
                                        hasNewlines = true;
                                    }
                                    
                                    if (fieldExamples.length < 5) {
                                        fieldExamples.push({
                                            name: fieldName,
                                            value: fieldValue.substring(0, 50)
                                        });
                                    }
                                    totalFieldCount++;
                                }
                            });
                        }
                        
                        // URLs supplémentaires
                        if (item.login?.uris && item.login.uris.length > 1) {
                            item.login.uris.slice(1).forEach((uri, index) => {
                                if (uri.uri) {
                                    customFields[`URL${index + 2}`] = processText(uri.uri);
                                }
                            });
                        }
                        
                        // Ajouter custom_fields seulement s'il y en a
                        if (Object.keys(customFields).length > 0) {
                            record.custom_fields = customFields;
                        }
                        
                        loginCount++;
                        
                    } else if (item.type === 2) {
                        // Type NOTE
                        record.type = "note";  // SANS $ pour les notes
                        
                        // Préserver les \n dans les notes
                        if (item.notes) {
                            record.notes = processText(item.notes, true);
                            if (item.notes.includes('\n')) {
                                hasNewlines = true;
                            }
                        } else {
                            record.notes = '';
                        }
                        
                        // Les notes peuvent avoir des champs personnalisés
                        if (item.fields && Array.isArray(item.fields)) {
                            const customFields = {};
                            item.fields.forEach(field => {
                                if (field.name && field.value !== undefined && field.value !== null) {
                                    const fieldName = processText(field.name);
                                    const fieldValue = processText(String(field.value), true);
                                    customFields[fieldName] = fieldValue;
                                    if (String(field.value).includes('\n')) {
                                        hasNewlines = true;
                                    }
                                    totalFieldCount++;
                                }
                            });
                            if (Object.keys(customFields).length > 0) {
                                record.custom_fields = customFields;
                            }
                        }
                        
                        noteCount++;
                    }
                    
                    // Format UNIFORME: folders est TOUJOURS un tableau d'objets
                    record.folders = [{
                        folder: folderPath
                    }];
                    
                    result.records.push(record);
                    itemCount++;
                });
            }
            
            keeperData = result;
            
            // Afficher l'avertissement si des \n ont été détectés
            if (hasNewlines) {
                document.getElementById('newlineWarning').style.display = 'block';
            }
            
            // Afficher les stats
            document.getElementById('stats').style.display = 'grid';
            document.getElementById('itemCount').textContent = itemCount;
            document.getElementById('folderCount').textContent = Object.keys({...folders, ...collections}).length;
            document.getElementById('fieldCount').textContent = totalFieldCount;
            
            // Afficher les exemples de champs
            if (fieldExamples.length > 0) {
                const fieldMapping = document.getElementById('fieldMapping');
                fieldMapping.style.display = 'block';
                const fieldList = document.getElementById('fieldList');
                fieldList.innerHTML = fieldExamples.map(f => {
                    const displayValue = f.value.replace(/\n/g, '\\n');
                    return `<div class="field-item">"${f.name}": "${displayValue}${f.value.length === 50 ? '...' : ''}"</div>`;
                }).join('');
            }
            
            // Afficher l'aperçu
            if (result.records.length > 0) {
                const preview = document.getElementById('preview');
                preview.style.display = 'block';
                
                // Chercher un exemple de note et un exemple de login
                const noteExample = result.records.find(r => r.type === 'note');
                const loginExample = result.records.find(r => r['$type'] === 'login');
                
                let previewContent = '';
                if (loginExample) {
                    previewContent += '// Exemple de LOGIN:\n' + JSON.stringify(loginExample, null, 2);
                }
                if (noteExample) {
                    if (previewContent) previewContent += '\n\n';
                    previewContent += '// Exemple de NOTE:\n' + JSON.stringify(noteExample, null, 2);
                }
                if (!previewContent) {
                    previewContent = JSON.stringify(result.records[0], null, 2);
                }
                
                preview.textContent = previewContent;
                
                // Log dans la console
                console.log('Conversion terminée:');
                console.log(`- ${loginCount} login(s)`);
                console.log(`- ${noteCount} note(s)`);
                console.log(`- Total: ${itemCount} entrée(s)`);
                console.log('Format vérifié: folders est un tableau pour TOUS les types');
            }
            
            // Préparer les téléchargements
            prepareDownloads();
            
            // Afficher le résultat
            document.getElementById('result').classList.add('show');
            let prefixMessage = '';
            if (cleanPrefix) {
                prefixMessage = `<br>📁 <strong>Préfixe appliqué :</strong> ${cleanPrefix}`;
            }
            
            let typeMessage = '';
            if (notesOnly) {
                typeMessage = `<br>📝 <strong>Export :</strong> ${noteCount} note(s) uniquement`;
            } else {
                typeMessage = `<br>🔐 <strong>Export :</strong> ${loginCount} login(s) + ${noteCount} note(s)`;
            }
            
            document.getElementById('resultMessage').innerHTML = 
                `✅ ${itemCount} entrée(s) converties${typeMessage}<br>` +
                `✅ ${Object.keys({...folders, ...collections}).length} dossier(s)/collection(s)<br>` +
                `✅ ${totalFieldCount} champ(s) personnalisé(s) converti(s)${prefixMessage}<br>` +
                `✅ Retours à la ligne (\\n) préservés<br>` +
                `✅ Format uniforme : folders est toujours [{folder: "..."}]`;
        }

        function prepareDownloads() {
            // JSON - Le JSON.stringify préserve automatiquement les \n
            const jsonBlob = new Blob([JSON.stringify(keeperData, null, 2)], {
                type: 'application/json'
            });
            document.getElementById('downloadJSON').href = URL.createObjectURL(jsonBlob);
            
            // Nom du fichier selon le type d'export
            const hasNotes = keeperData.records.some(r => r.type === 'note');
            const hasLogins = keeperData.records.some(r => r['$type'] === 'login');
            let filename = 'keeper_import';
            if (hasNotes && !hasLogins) {
                filename = 'keeper_notes_only';
            } else if (hasLogins && !hasNotes) {
                filename = 'keeper_logins_only';
            }
            
            document.getElementById('downloadJSON').download = filename + '.json';
            
            // CSV
            const csv = convertToCSV();
            const csvBlob = new Blob(['\ufeff' + csv], {
                type: 'text/csv;charset=utf-8'
            });
            document.getElementById('downloadCSV').href = URL.createObjectURL(csvBlob);
            document.getElementById('downloadCSV').download = filename + '.csv';
        }

        function convertToCSV() {
            const headers = ['Folder', 'Title', 'Login', 'Password', 'Website Address', 'Notes', 'Shared Folder', 'Custom Fields'];
            let csv = headers.map(h => `"${h}"`).join(',') + '\n';
            
            keeperData.records.forEach(record => {
                // Maintenant TOUS les types utilisent folders[0].folder
                const folder = record.folders?.[0]?.folder || '';
                
                // Formatter les custom fields pour CSV
                let customFieldsStr = '';
                if (record.custom_fields) {
                    const fields = [];
                    for (const [key, val] of Object.entries(record.custom_fields)) {
                        fields.push(`${key}:${val}`);
                    }
                    customFieldsStr = fields.join(' | ');
                }
                
                const row = [
                    folder,
                    record.title,
                    record.login || '',
                    record.password || '',
                    record.login_url || '',
                    record.notes || '',
                    '', // Shared folder vide
                    customFieldsStr
                ];
                
                csv += row.map(f => `"${String(f).replace(/"/g, '""')}"`).join(',') + '\n';
            });
            
            return csv;
        }

        function showError(message) {
            document.getElementById('result').classList.add('show', 'error');
            document.getElementById('resultTitle').textContent = '❌ Erreur';
            document.getElementById('resultMessage').textContent = message;
            document.querySelector('#result div').style.display = 'none';
        }
    </script>
</body>
</html>
