<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Nettoyeur de doublons - Évaluation des compétences</title>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            justify-content: center;
            align-items: center;
            padding: 20px;
        }
        
        .container {
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            padding: 40px;
            max-width: 600px;
            width: 100%;
        }
        
        h1 {
            color: #333;
            margin-bottom: 10px;
            font-size: 28px;
            text-align: center;
        }
        
        .subtitle {
            color: #666;
            text-align: center;
            margin-bottom: 30px;
            font-size: 14px;
        }
        
        .file-input-wrapper {
            position: relative;
            overflow: hidden;
            display: inline-block;
            width: 100%;
        }
        
        .file-input-label {
            display: block;
            padding: 15px 30px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 10px;
            cursor: pointer;
            text-align: center;
            font-weight: 600;
            transition: transform 0.2s;
        }
        
        .file-input-label:hover {
            transform: translateY(-2px);
        }
        
        #fileInput {
            position: absolute;
            left: -9999px;
        }
        
        .status {
            margin: 20px 0;
            padding: 15px;
            border-radius: 10px;
            display: none;
        }
        
        .status.info {
            background: #e3f2fd;
            color: #1976d2;
            border: 1px solid #90caf9;
        }
        
        .status.success {
            background: #e8f5e9;
            color: #2e7d32;
            border: 1px solid #81c784;
        }
        
        .status.error {
            background: #ffebee;
            color: #c62828;
            border: 1px solid #ef5350;
        }
        
        .stats {
            display: none;
            margin: 20px 0;
            padding: 20px;
            background: #f5f5f5;
            border-radius: 10px;
        }
        
        .stat-item {
            display: flex;
            justify-content: space-between;
            margin-bottom: 10px;
            padding-bottom: 10px;
            border-bottom: 1px solid #ddd;
        }
        
        .stat-item:last-child {
            border-bottom: none;
            margin-bottom: 0;
            padding-bottom: 0;
        }
        
        .stat-label {
            color: #666;
        }
        
        .stat-value {
            font-weight: bold;
            color: #333;
        }
        
        .download-btn {
            display: none;
            width: 100%;
            padding: 15px;
            background: linear-gradient(135deg, #4caf50 0%, #45a049 100%);
            color: white;
            border: none;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: transform 0.2s;
            margin-top: 20px;
        }
        
        .download-btn:hover {
            transform: translateY(-2px);
        }
        
        .duplicates-list {
            max-height: 200px;
            overflow-y: auto;
            margin-top: 10px;
            padding: 10px;
            background: white;
            border-radius: 8px;
            font-size: 14px;
            display: none;
        }
        
        .duplicate-item {
            padding: 5px 0;
            border-bottom: 1px solid #f0f0f0;
            color: #666;
        }
        
        .duplicate-item:last-child {
            border-bottom: none;
        }
        
        .toggle-duplicates {
            color: #667eea;
            cursor: pointer;
            text-decoration: underline;
            font-size: 14px;
            margin-top: 10px;
            display: inline-block;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🧹 Nettoyeur de doublons Excel</h1>
        <p class="subtitle">Supprimez automatiquement les compétences en double</p>
        
        <div class="file-input-wrapper">
            <label for="fileInput" class="file-input-label">
                📁 Sélectionner votre fichier Excel
            </label>
            <input type="file" id="fileInput" accept=".xlsx,.xls" />
        </div>
        
        <div id="status" class="status"></div>
        
        <div id="stats" class="stats">
            <h3 style="margin-bottom: 15px; color: #333;">📊 Résultats de l'analyse</h3>
            <div class="stat-item">
                <span class="stat-label">Lignes originales :</span>
                <span class="stat-value" id="originalRows">0</span>
            </div>
            <div class="stat-item">
                <span class="stat-label">Doublons détectés :</span>
                <span class="stat-value" id="duplicatesFound" style="color: #f44336;">0</span>
            </div>
            <div class="stat-item">
                <span class="stat-label">Lignes après nettoyage :</span>
                <span class="stat-value" id="cleanedRows" style="color: #4caf50;">0</span>
            </div>
            <span class="toggle-duplicates" onclick="toggleDuplicatesList()">Voir les doublons supprimés</span>
            <div id="duplicatesList" class="duplicates-list"></div>
        </div>
        
        <button id="downloadBtn" class="download-btn" onclick="downloadCleanedFile()">
            💾 Télécharger le fichier nettoyé
        </button>
    </div>

    <script>
        let cleanedWorkbook = null;
        let originalFilename = '';
        
        document.getElementById('fileInput').addEventListener('change', handleFile);
        
        function handleFile(e) {
            const file = e.target.files[0];
            if (!file) return;
            
            originalFilename = file.name.replace(/\.[^/.]+$/, "");
            showStatus('info', '⏳ Analyse du fichier en cours...');
            
            const reader = new FileReader();
            reader.onload = function(e) {
                try {
                    const data = new Uint8Array(e.target.result);
                    const workbook = XLSX.read(data, {type: 'array'});
                    
                    processWorkbook(workbook);
                } catch (error) {
                    showStatus('error', '❌ Erreur lors de la lecture du fichier');
                    console.error(error);
                }
            };
            reader.readAsArrayBuffer(file);
        }
        
        function processWorkbook(workbook) {
            const sheetName = workbook.SheetNames[0];
            const sheet = workbook.Sheets[sheetName];
            const data = XLSX.utils.sheet_to_json(sheet, {header: 1, defval: ""});
            
            const originalCount = data.length;
            const seen = new Map();
            const cleanedData = [];
            const duplicatesList = [];
            
            // Traiter chaque ligne
            for (let i = 0; i < data.length; i++) {
                const row = data[i];
                const firstCell = row[0] ? row[0].toString().trim() : "";
                
                // Si c'est une ligne vide ou un titre sans contenu dans la première colonne, on la garde
                if (firstCell === "" || !seen.has(firstCell)) {
                    cleanedData.push(row);
                    if (firstCell !== "") {
                        seen.set(firstCell, {
                            firstRow: i + 1,
                            count: 1
                        });
                    }
                } else {
                    // C'est un doublon
                    const existing = seen.get(firstCell);
                    existing.count++;
                    duplicatesList.push({
                        text: firstCell,
                        rowNum: i + 1,
                        firstOccurrence: existing.firstRow
                    });
                }
            }
            
            // Créer le nouveau workbook
            const newWorkbook = XLSX.utils.book_new();
            const newSheet = XLSX.utils.aoa_to_sheet(cleanedData);
            
            // Copier les propriétés de formatage si elles existent
            if (sheet['!cols']) newSheet['!cols'] = sheet['!cols'];
            if (sheet['!rows']) newSheet['!rows'] = sheet['!rows'];
            if (sheet['!merges']) newSheet['!merges'] = sheet['!merges'];
            
            XLSX.utils.book_append_sheet(newWorkbook, newSheet, sheetName);
            cleanedWorkbook = newWorkbook;
            
            // Afficher les statistiques
            showStats(originalCount, duplicatesList.length, cleanedData.length, duplicatesList);
            showStatus('success', '✅ Fichier nettoyé avec succès !');
            document.getElementById('downloadBtn').style.display = 'block';
        }
        
        function showStatus(type, message) {
            const statusEl = document.getElementById('status');
            statusEl.className = 'status ' + type;
            statusEl.textContent = message;
            statusEl.style.display = 'block';
        }
        
        function showStats(original, duplicates, cleaned, duplicatesList) {
            document.getElementById('originalRows').textContent = original;
            document.getElementById('duplicatesFound').textContent = duplicates;
            document.getElementById('cleanedRows').textContent = cleaned;
            document.getElementById('stats').style.display = 'block';
            
            // Remplir la liste des doublons
            const listEl = document.getElementById('duplicatesList');
            listEl.innerHTML = '';
            duplicatesList.slice(0, 50).forEach(dup => {
                const div = document.createElement('div');
                div.className = 'duplicate-item';
                div.textContent = `Ligne ${dup.rowNum}: "${dup.text.substring(0, 50)}${dup.text.length > 50 ? '...' : ''}" (première occurrence: ligne ${dup.firstOccurrence})`;
                listEl.appendChild(div);
            });
            
            if (duplicatesList.length > 50) {
                const div = document.createElement('div');
                div.className = 'duplicate-item';
                div.style.fontStyle = 'italic';
                div.textContent = `... et ${duplicatesList.length - 50} autres doublons`;
                listEl.appendChild(div);
            }
        }
        
        function toggleDuplicatesList() {
            const list = document.getElementById('duplicatesList');
            list.style.display = list.style.display === 'none' || list.style.display === '' ? 'block' : 'none';
        }
        
        function downloadCleanedFile() {
            if (!cleanedWorkbook) return;
            
            const wbout = XLSX.write(cleanedWorkbook, {bookType: 'xlsx', type: 'binary'});
            const blob = new Blob([s2ab(wbout)], {type: 'application/octet-stream'});
            const url = URL.createObjectURL(blob);
            
            const a = document.createElement('a');
            a.href = url;
            a.download = originalFilename + '_nettoyé.xlsx';
            a.click();
            
            URL.revokeObjectURL(url);
        }
        
        function s2ab(s) {
            const buf = new ArrayBuffer(s.length);
            const view = new Uint8Array(buf);
            for (let i = 0; i < s.length; i++) {
                view[i] = s.charCodeAt(i) & 0xFF;
            }
            return buf;
        }
        
        // Auto-process if file is already loaded
        window.addEventListener('DOMContentLoaded', async () => {
            try {
                const response = await window.fs.readFile('évaluation des compétences.xlsx');
                const workbook = XLSX.read(response, {type: 'array'});
                originalFilename = 'évaluation des compétences';
                processWorkbook(workbook);
            } catch (e) {
                // Le fichier n'est pas disponible dans le contexte, attendre le téléchargement manuel
            }
        });
    </script>
</body>
</html>
