<?php
/**
 * Configuration du jeu Undercover
 */

// Démarrer la session
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Répertoire pour stocker les parties
define('GAMES_DIR', __DIR__ . '/games/');

// Créer le répertoire s'il n'existe pas
if (!is_dir(GAMES_DIR)) {
    if (!mkdir(GAMES_DIR, 0777, true)) {
        die('Erreur: Impossible de créer le dossier games/. Vérifiez les permissions.');
    }
    chmod(GAMES_DIR, 0777);
}

// Vérifier que le dossier est writable
if (!is_writable(GAMES_DIR)) {
    chmod(GAMES_DIR, 0777);
    if (!is_writable(GAMES_DIR)) {
        die('Erreur: Le dossier games/ n\'est pas accessible en écriture.');
    }
}

// Paires de mots pour le jeu (mot civil, mot undercover)
$WORD_PAIRS = [
    // ===== RÉSEAUX SOCIAUX & TECH =====
    ['TikTok', 'Instagram Reels'],
    ['Snapchat', 'BeReal'],
    ['Twitter/X', 'Threads'],
    ['Discord', 'Slack'],
    ['Tinder', 'Bumble'],
    ['OnlyFans', 'Patreon'],
    ['Twitch', 'YouTube Gaming'],
    ['ChatGPT', 'Google Gemini'],
    ['iPhone', 'Samsung Galaxy'],
    ['AirPods', 'Galaxy Buds'],
    ['Netflix', 'Disney+'],
    ['Spotify', 'Apple Music'],
    ['Uber', 'Bolt'],
    ['Deliveroo', 'Uber Eats'],
    ['PayPal', 'Lydia'],
    ['Vinted', 'Leboncoin'],
    ['Amazon', 'Aliexpress'],
    ['Google', 'Bing'],
    ['WhatsApp', 'Telegram'],
    ['LinkedIn', 'Indeed'],
    
    // ===== MUSIQUE & ARTISTES =====
    ['Jul', 'SCH'],
    ['Ninho', 'Gazo'],
    ['Aya Nakamura', 'Wejdene'],
    ['PNL', 'Laylow'],
    ['Nekfeu', 'Orelsan'],
    ['Booba', 'Kaaris'],
    ['Dua Lipa', 'Aya Nakamura'],
    ['Taylor Swift', 'Beyoncé'],
    ['Drake', 'Kendrick Lamar'],
    ['Travis Scott', 'Playboi Carti'],
    ['Bad Bunny', 'J Balvin'],
    ['The Weeknd', 'Bruno Mars'],
    ['Rihanna', 'Beyoncé'],
    ['Ed Sheeran', 'Shawn Mendes'],
    ['Billie Eilish', 'Olivia Rodrigo'],
    ['Rap', 'Trap'],
    ['Drill', 'Afrotrap'],
    ['Reggaeton', 'Afrobeat'],
    ['K-Pop', 'J-Pop'],
    ['Festival', 'Concert'],
    
    // ===== SÉRIES & FILMS =====
    ['Squid Game', 'Alice in Borderland'],
    ['Stranger Things', 'Dark'],
    ['Money Heist', 'Lupin'],
    ['Euphoria', 'Skins'],
    ['The Office', 'Parks and Recreation'],
    ['Breaking Bad', 'Better Call Saul'],
    ['Game of Thrones', 'House of the Dragon'],
    ['Wednesday', 'Mercredi'],
    ['Emily in Paris', 'Sex and the City'],
    ['Naruto', 'One Piece'],
    ['Dragon Ball', 'One Punch Man'],
    ['Attack on Titan', 'Demon Slayer'],
    ['Marvel', 'DC Comics'],
    ['Spider-Man', 'Batman'],
    ['Avengers', 'Justice League'],
    ['Barbie', 'Oppenheimer'],
    ['Avatar', 'Titanic'],
    ['Star Wars', 'Star Trek'],
    ['Harry Potter', 'Le Seigneur des Anneaux'],
    ['Disney', 'Pixar'],
    
    // ===== GAMING =====
    ['Fortnite', 'Apex Legends'],
    ['FIFA/FC', 'PES/eFootball'],
    ['GTA', 'Red Dead Redemption'],
    ['Call of Duty', 'Battlefield'],
    ['Minecraft', 'Roblox'],
    ['League of Legends', 'Dota 2'],
    ['Valorant', 'CS2'],
    ['Elden Ring', 'Dark Souls'],
    ['Zelda', 'Genshin Impact'],
    ['PlayStation', 'Xbox'],
    ['Nintendo Switch', 'Steam Deck'],
    ['Pokémon', 'Digimon'],
    ['Mario', 'Sonic'],
    ['Among Us', 'Fall Guys'],
    ['Candy Crush', 'Subway Surfers'],
    
    // ===== SEXUALITÉ & RELATIONS =====
    ['Crush', 'Béguin'],
    ['Ex', 'Plan cul'],
    ['Friendzone', 'Lovezone'],
    ['Ghosting', 'Breadcrumbing'],
    ['Red flag', 'Green flag'],
    ['Situationship', 'Relation libre'],
    ['DTF', 'Netflix and chill'],
    ['Nude', 'Dick pic'],
    ['Sextape', 'Leak'],
    ['Capote', 'Pilule'],
    ['Viagra', 'Cialis'],
    ['Vibro', 'Sextoy'],
    ['Menottes', 'Bandeau'],
    ['Fétichisme', 'Fantasme'],
    ['Polyamour', 'Couple libre'],
    ['Coming out', 'Outing'],
    ['Gay', 'Bi'],
    ['Lesbienne', 'Queer'],
    ['Trans', 'Non-binaire'],
    ['Drag queen', 'Drag king'],
    
    // ===== TENDANCES & LIFESTYLE =====
    ['Influenceur', 'Créateur de contenu'],
    ['Viral', 'Buzz'],
    ['Cancel culture', 'Woke'],
    ['Boomer', 'Zoomer'],
    ['Millennial', 'Gen Z'],
    ['Cringe', 'Based'],
    ['Slay', 'Ate'],
    ['NPC', 'Main character'],
    ['GOAT', 'WOAT'],
    ['Ratio', 'L + ratio'],
    ['Vegan', 'Végétarien'],
    ['Crossfit', 'Musculation'],
    ['Yoga', 'Pilates'],
    ['Méditation', 'Mindfulness'],
    ['CBD', 'THC'],
    ['Chicha', 'Cigarette électronique'],
    ['Apéro', 'Pré-soirée'],
    ['Brunch', 'Petit-déj'],
    ['Poke bowl', 'Buddha bowl'],
    ['Matcha', 'Bubble tea'],
    
    // ===== ARGOT & EXPRESSIONS =====
    ['Frère', 'Bro'],
    ['Wesh', 'Yo'],
    ['Bail', 'Truc'],
    ['Chanmé', 'Ouf'],
    ['Daron', 'Daronne'],
    ['Ken', 'Pécho'],
    ['Boloss', 'Tocard'],
    ['Seum', 'Rage'],
    ['Darty', 'Fnac'],
    ['Cramé', 'Grillé'],
    ['Chiller', 'Poser'],
    ['Galère', 'Merde'],
    ['Thune', 'Oseille'],
    ['Taf', 'Job'],
    ['Soirée', 'Teuf'],
    ['Beuh', 'Weed'],
    ['Coke', 'Speed'],
    ['Défoncé', 'Bourré'],
    ['Gueule de bois', 'Lendemain difficile'],
    ['Clope', 'Cigarette'],
    
    // ===== ACTUALITÉ & POLITIQUE =====
    ['Macron', 'Le Pen'],
    ['Gauche', 'Droite'],
    ['Grève', 'Manifestation'],
    ['Gilets jaunes', 'Black bloc'],
    ['Retraite', 'Chômage'],
    ['Inflation', 'Récession'],
    ['RSA', 'Pôle emploi'],
    ['SMIC', 'Salaire minimum'],
    ['Loyer', 'Proprio'],
    ['Ukraine', 'Russie'],
    ['Gaza', 'Israël'],
    ['Trump', 'Biden'],
    ['Brexit', 'Union européenne'],
    ['Covid', 'Grippe'],
    ['Vaccin', 'Pfizer'],
    ['Confinement', 'Couvre-feu'],
    ['Télétravail', 'Présentiel'],
    ['IA', 'Robot'],
    ['Climat', 'Écologie'],
    ['Vélo', 'Trottinette électrique'],
    
    // ===== SPORT =====
    ['PSG', 'OM'],
    ['Real Madrid', 'Barcelone'],
    ['Mbappé', 'Haaland'],
    ['Messi', 'Ronaldo'],
    ['Zidane', 'Henry'],
    ['Champions League', 'Ligue 1'],
    ['Coupe du monde', 'Euro'],
    ['JO Paris', 'JO Tokyo'],
    ['MMA', 'Boxe'],
    ['NBA', 'Euroleague'],
    ['Tennis', 'Padel'],
    ['F1', 'MotoGP'],
    ['Tour de France', 'Giro'],
    ['Rugby', 'Football américain'],
    ['Surf', 'Skate'],
    
    // ===== BOUFFE & DRINK =====
    ['McDo', 'Burger King'],
    ['KFC', 'Popeyes'],
    ['Starbucks', 'Columbus Café'],
    ['Kebab', 'Tacos'],
    ['Sushi', 'Poke bowl'],
    ['Pizza', 'Pasta'],
    ['Croissant', 'Pain au chocolat'],
    ['Nutella', 'Beurre de cacahuète'],
    ['Red Bull', 'Monster'],
    ['Coca', 'Pepsi'],
    ['Bière', 'Vin'],
    ['Vodka', 'Whisky'],
    ['Mojito', 'Spritz'],
    ['Champagne', 'Prosecco'],
    ['Perrier', 'San Pellegrino'],
    
    // ===== MODE & MARQUES =====
    ['Nike', 'Adidas'],
    ['Jordan', 'Yeezy'],
    ['Supreme', 'Palace'],
    ['Zara', 'H&M'],
    ['Shein', 'Primark'],
    ['Louis Vuitton', 'Gucci'],
    ['Balenciaga', 'Off-White'],
    ['Lacoste', 'Ralph Lauren'],
    ['The North Face', 'Patagonia'],
    ['Ray-Ban', 'Oakley'],
    ['Rolex', 'Omega'],
    ['Crocs', 'Birkenstock'],
    ['New Balance', 'Asics'],
    ['Converse', 'Vans'],
    ['Levis', 'Diesel'],
    
    // ===== CLASSIQUES REVISITÉS =====
    ['Chat', 'Chien'],
    ['Pizza', 'Burger'],
    ['Café', 'Thé'],
    ['Plage', 'Montagne'],
    ['Été', 'Hiver'],
    ['Matin', 'Soir'],
    ['Ville', 'Campagne'],
    ['Voiture', 'Moto'],
    ['Avion', 'Train'],
    ['Livre', 'Film'],
];

/**
 * Génère un code de partie aléatoire
 */
function generateGameCode(): string {
    $chars = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
    $code = '';
    for ($i = 0; $i < 4; $i++) {
        $code .= $chars[random_int(0, strlen($chars) - 1)];
    }
    return $code;
}

/**
 * Récupère le chemin du fichier d'une partie
 */
function getGameFilePath(string $code): string {
    return GAMES_DIR . strtoupper($code) . '.json';
}

/**
 * Charge une partie depuis le fichier JSON
 */
function loadGame(string $code): ?array {
    $path = getGameFilePath($code);
    if (!file_exists($path)) {
        return null;
    }
    $content = file_get_contents($path);
    return json_decode($content, true);
}

/**
 * Sauvegarde une partie dans le fichier JSON
 */
function saveGame(string $code, array $game): bool {
    $path = getGameFilePath($code);
    return file_put_contents($path, json_encode($game, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)) !== false;
}

/**
 * Supprime les parties anciennes (plus de 24h)
 */
function cleanOldGames(): void {
    $files = glob(GAMES_DIR . '*.json');
    $now = time();
    foreach ($files as $file) {
        if ($now - filemtime($file) > 86400) { // 24 heures
            unlink($file);
        }
    }
}

/**
 * Calcule le nombre de joueurs par rôle
 */
function calculateRoles(int $totalPlayers): array {
    if ($totalPlayers < 3) {
        return ['civilians' => $totalPlayers, 'undercovers' => 0, 'mrwhites' => 0];
    }
    
    $undercovers = max(1, floor($totalPlayers / 4));
    $mrwhites = ($totalPlayers >= 6) ? 1 : 0;
    $civilians = $totalPlayers - $undercovers - $mrwhites;
    
    return [
        'civilians' => $civilians,
        'undercovers' => $undercovers,
        'mrwhites' => $mrwhites
    ];
}

/**
 * Assigne les rôles aux joueurs
 */
function assignRoles(array $players, array $wordPair): array {
    global $WORD_PAIRS;
    
    $totalPlayers = count($players);
    $roles = calculateRoles($totalPlayers);
    
    // Mélanger les joueurs
    $playerIds = array_keys($players);
    shuffle($playerIds);
    
    $index = 0;
    
    // Assigner les civils
    for ($i = 0; $i < $roles['civilians']; $i++) {
        $players[$playerIds[$index]]['role'] = 'civilian';
        $players[$playerIds[$index]]['word'] = $wordPair[0];
        $index++;
    }
    
    // Assigner les undercovers
    for ($i = 0; $i < $roles['undercovers']; $i++) {
        $players[$playerIds[$index]]['role'] = 'undercover';
        $players[$playerIds[$index]]['word'] = $wordPair[1];
        $index++;
    }
    
    // Assigner les Mr. White
    for ($i = 0; $i < $roles['mrwhites']; $i++) {
        $players[$playerIds[$index]]['role'] = 'mrwhite';
        $players[$playerIds[$index]]['word'] = null;
        $index++;
    }
    
    return $players;
}

/**
 * Vérifie les conditions de victoire
 */
function checkVictory(array $players): ?string {
    $aliveCivilians = 0;
    $aliveUndercovers = 0;
    $aliveMrWhites = 0;
    
    foreach ($players as $player) {
        if ($player['alive']) {
            switch ($player['role']) {
                case 'civilian':
                    $aliveCivilians++;
                    break;
                case 'undercover':
                    $aliveUndercovers++;
                    break;
                case 'mrwhite':
                    $aliveMrWhites++;
                    break;
            }
        }
    }
    
    $aliveInfiltrators = $aliveUndercovers + $aliveMrWhites;
    
    // Victoire des civils : tous les infiltrés sont éliminés
    if ($aliveInfiltrators === 0) {
        return 'civilians';
    }
    
    // Victoire des infiltrés : il reste 1 civil ou moins
    if ($aliveCivilians <= 1 && $aliveInfiltrators > 0) {
        return 'infiltrators';
    }
    
    return null;
}

// Nettoyer les anciennes parties périodiquement
if (random_int(1, 10) === 1) {
    cleanOldGames();
}
