<?php
require_once __DIR__ . '/config.php';

header('Content-Type: application/json');

// Fonction de réponse JSON
function jsonResponse($data) {
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

function jsonError($message) {
    jsonResponse(['error' => $message]);
}

// Récupérer les paramètres
$action = $_REQUEST['action'] ?? '';
$code = strtoupper($_REQUEST['code'] ?? '');
$playerId = $_REQUEST['player'] ?? '';

// Actions qui ne nécessitent pas de partie existante
if ($action === 'status') {
    if (!$code || !$playerId) {
        jsonError('missing_params');
    }
    
    $game = loadGame($code);
    
    if (!$game) {
        jsonError('game_not_found');
    }
    
    if (!isset($game['players'][$playerId])) {
        jsonError('player_not_found');
    }
    
    $player = $game['players'][$playerId];
    
    // Préparer les données des joueurs (sans révéler les rôles sauf si terminé)
    $playersData = [];
    foreach ($game['players'] as $pid => $p) {
        $playerData = [
            'id' => $pid,
            'name' => $p['name'],
            'alive' => $p['alive'],
            'hasDescribed' => $p['hasDescribed'] ?? false,
            'votes' => $p['votes'] ?? 0,
            'isHost' => ($game['host'] === $pid)
        ];
        
        // Révéler les rôles seulement si la partie est terminée
        if ($game['phase'] === 'ended' || $game['phase'] === 'reveal') {
            $playerData['role'] = $p['role'];
            $playerData['word'] = $p['word'];
        }
        
        $playersData[] = $playerData;
    }
    
    // Ordre de description pour la phase description
    $descriptionOrder = [];
    if (isset($game['descriptionOrder']) && !empty($game['descriptionOrder'])) {
        foreach ($game['descriptionOrder'] as $pid) {
            if (isset($game['players'][$pid])) {
                $p = $game['players'][$pid];
                $descriptionOrder[] = [
                    'id' => $pid,
                    'name' => $p['name'],
                    'alive' => $p['alive'],
                    'hasDescribed' => $p['hasDescribed'] ?? false
                ];
            }
        }
    }
    
    // Calculer les statistiques
    $aliveCount = 0;
    $votedCount = 0;
    foreach ($game['players'] as $p) {
        if ($p['alive']) {
            $aliveCount++;
            if (!empty($p['votedFor'])) {
                $votedCount++;
            }
        }
    }
    
    // Données du joueur éliminé
    $eliminatedPlayerData = null;
    if (isset($game['eliminatedPlayer']) && isset($game['players'][$game['eliminatedPlayer']])) {
        $ep = $game['players'][$game['eliminatedPlayer']];
        $eliminatedPlayerData = [
            'id' => $game['eliminatedPlayer'],
            'name' => $ep['name'],
            'role' => $ep['role']
        ];
    }
    
    // Vérifier si c'est le tour du joueur actuel
    $isCurrentDescriber = false;
    if ($game['phase'] === 'description' && !empty($game['descriptionOrder'])) {
        $currentIndex = $game['currentDescriber'] ?? 0;
        if (isset($game['descriptionOrder'][$currentIndex])) {
            $isCurrentDescriber = ($game['descriptionOrder'][$currentIndex] === $playerId);
        }
    }
    
    $response = [
        'code' => $code,
        'status' => $game['status'],
        'phase' => $game['phase'],
        'round' => $game['round'],
        'playerCount' => count($game['players']),
        'aliveCount' => $aliveCount,
        'votedCount' => $votedCount,
        'allVoted' => ($votedCount === $aliveCount),
        'isHost' => ($game['host'] === $playerId),
        'player' => [
            'id' => $playerId,
            'name' => $player['name'],
            'role' => $player['role'],
            'word' => $player['word'],
            'alive' => $player['alive'],
            'hasDescribed' => $player['hasDescribed'] ?? false,
            'votedFor' => $player['votedFor'] ?? null
        ],
        'players' => $playersData,
        'descriptionOrder' => $descriptionOrder,
        'currentDescriber' => $game['currentDescriber'] ?? 0,
        'isCurrentDescriber' => $isCurrentDescriber,
        'eliminatedPlayer' => $eliminatedPlayerData,
        'winner' => $game['winner'],
        'roleDistribution' => calculateRoles(count($game['players']))
    ];
    
    // Ajouter les mots si la partie est terminée
    if ($game['phase'] === 'ended' && isset($game['wordPair'])) {
        $response['civilianWord'] = $game['wordPair'][0];
        $response['undercoverWord'] = $game['wordPair'][1];
    }
    
    // Ajouter la tentative de Mr. White
    if (isset($game['mrWhiteGuess'])) {
        $response['mrWhiteGuess'] = $game['mrWhiteGuess'];
    }
    
    jsonResponse($response);
}

// Les autres actions nécessitent une partie et un joueur valides
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonError('method_not_allowed');
}

if (!$code || !$playerId) {
    jsonError('missing_params');
}

$game = loadGame($code);

if (!$game) {
    jsonError('Partie non trouvée');
}

if (!isset($game['players'][$playerId])) {
    jsonError('Joueur non trouvé');
}

$player = $game['players'][$playerId];
$isHost = ($game['host'] === $playerId);

switch ($action) {
    case 'start':
        if (!$isHost) {
            jsonError('Seul l\'hôte peut lancer la partie');
        }
        
        if ($game['status'] !== 'waiting') {
            jsonError('La partie a déjà commencé');
        }
        
        if (count($game['players']) < 3) {
            jsonError('Il faut au moins 3 joueurs');
        }
        
        // Choisir une paire de mots aléatoire
        global $WORD_PAIRS;
        $wordPair = $WORD_PAIRS[array_rand($WORD_PAIRS)];
        
        // Mélanger pour que parfois les civils aient le second mot
        if (random_int(0, 1) === 1) {
            $wordPair = array_reverse($wordPair);
        }
        
        $game['wordPair'] = $wordPair;
        
        // Assigner les rôles
        $game['players'] = assignRoles($game['players'], $wordPair);
        
        // Créer l'ordre de description (joueurs vivants mélangés)
        $alivePlayers = array_keys(array_filter($game['players'], fn($p) => $p['alive']));
        shuffle($alivePlayers);
        $game['descriptionOrder'] = $alivePlayers;
        
        // Passer en mode jeu
        $game['status'] = 'playing';
        $game['phase'] = 'description';
        $game['round'] = 1;
        $game['currentDescriber'] = 0;
        $game['updatedAt'] = time();
        
        saveGame($code, $game);
        jsonResponse(['success' => true]);
        break;
        
    case 'described':
        if ($game['phase'] !== 'description') {
            jsonError('Ce n\'est pas la phase de description');
        }
        
        if (!$player['alive']) {
            jsonError('Vous êtes éliminé');
        }
        
        // Vérifier que c'est bien le tour de ce joueur
        $currentIndex = $game['currentDescriber'] ?? 0;
        if (!isset($game['descriptionOrder'][$currentIndex]) || 
            $game['descriptionOrder'][$currentIndex] !== $playerId) {
            jsonError('Ce n\'est pas votre tour');
        }
        
        // Marquer comme ayant décrit
        $game['players'][$playerId]['hasDescribed'] = true;
        
        // Passer au joueur suivant
        $game['currentDescriber']++;
        
        // Trouver le prochain joueur vivant qui n'a pas décrit
        while ($game['currentDescriber'] < count($game['descriptionOrder'])) {
            $nextId = $game['descriptionOrder'][$game['currentDescriber']];
            if (isset($game['players'][$nextId]) && 
                $game['players'][$nextId]['alive'] && 
                !$game['players'][$nextId]['hasDescribed']) {
                break;
            }
            $game['currentDescriber']++;
        }
        
        // Si tous ont décrit, passer à la discussion
        if ($game['currentDescriber'] >= count($game['descriptionOrder'])) {
            $game['phase'] = 'discussion';
        }
        
        $game['updatedAt'] = time();
        saveGame($code, $game);
        jsonResponse(['success' => true]);
        break;
        
    case 'start_voting':
        if (!$isHost) {
            jsonError('Seul l\'hôte peut lancer le vote');
        }
        
        if ($game['phase'] !== 'discussion') {
            jsonError('Ce n\'est pas la phase de discussion');
        }
        
        // Réinitialiser les votes
        foreach ($game['players'] as &$p) {
            $p['votes'] = 0;
            $p['votedFor'] = null;
        }
        unset($p);
        
        $game['phase'] = 'voting';
        $game['updatedAt'] = time();
        
        saveGame($code, $game);
        jsonResponse(['success' => true]);
        break;
        
    case 'vote':
        if ($game['phase'] !== 'voting') {
            jsonError('Ce n\'est pas la phase de vote');
        }
        
        if (!$player['alive']) {
            jsonError('Vous êtes éliminé et ne pouvez pas voter');
        }
        
        if (!empty($player['votedFor'])) {
            jsonError('Vous avez déjà voté');
        }
        
        $targetId = $_POST['target'] ?? '';
        
        // Option "passer" son vote
        if ($targetId === 'skip') {
            $game['players'][$playerId]['votedFor'] = 'skip';
            $game['updatedAt'] = time();
            saveGame($code, $game);
            jsonResponse(['success' => true]);
            break;
        }
        
        if (!isset($game['players'][$targetId])) {
            jsonError('Joueur cible non trouvé');
        }
        
        if (!$game['players'][$targetId]['alive']) {
            jsonError('Ce joueur est déjà éliminé');
        }
        
        if ($targetId === $playerId) {
            jsonError('Vous ne pouvez pas voter pour vous-même');
        }
        
        // Enregistrer le vote
        $game['players'][$playerId]['votedFor'] = $targetId;
        $game['players'][$targetId]['votes']++;
        $game['updatedAt'] = time();
        
        saveGame($code, $game);
        jsonResponse(['success' => true]);
        break;
        
    case 'end_voting':
        if (!$isHost) {
            jsonError('Seul l\'hôte peut terminer le vote');
        }
        
        if ($game['phase'] !== 'voting') {
            jsonError('Ce n\'est pas la phase de vote');
        }
        
        // Trouver le joueur avec le plus de votes
        $maxVotes = 0;
        $eliminatedId = null;
        $tiedPlayers = [];
        
        foreach ($game['players'] as $pid => $p) {
            if ($p['alive'] && $p['votes'] > 0) {
                if ($p['votes'] > $maxVotes) {
                    $maxVotes = $p['votes'];
                    $eliminatedId = $pid;
                    $tiedPlayers = [$pid];
                } elseif ($p['votes'] === $maxVotes) {
                    $tiedPlayers[] = $pid;
                }
            }
        }
        
        // En cas d'égalité, choisir aléatoirement
        if (count($tiedPlayers) > 1) {
            $eliminatedId = $tiedPlayers[array_rand($tiedPlayers)];
        }
        
        if ($eliminatedId) {
            $eliminatedPlayer = $game['players'][$eliminatedId];
            $game['eliminatedPlayer'] = $eliminatedId;
            
            // Si c'est Mr. White, lui donner une chance de deviner
            if ($eliminatedPlayer['role'] === 'mrwhite') {
                $game['phase'] = 'mrwhite_guess';
            } else {
                // Éliminer le joueur
                $game['players'][$eliminatedId]['alive'] = false;
                
                // Vérifier la victoire
                $winner = checkVictory($game['players']);
                
                if ($winner) {
                    $game['winner'] = $winner;
                    $game['phase'] = 'ended';
                } else {
                    $game['phase'] = 'reveal';
                }
            }
        } else {
            // Aucun vote, passer directement au tour suivant
            $game['phase'] = 'reveal';
        }
        
        $game['updatedAt'] = time();
        saveGame($code, $game);
        jsonResponse(['success' => true]);
        break;
        
    case 'mrwhite_guess':
        if ($game['phase'] !== 'mrwhite_guess') {
            jsonError('Ce n\'est pas le moment de deviner');
        }
        
        if ($game['eliminatedPlayer'] !== $playerId) {
            jsonError('Seul Mr. White éliminé peut deviner');
        }
        
        $guess = trim($_POST['guess'] ?? '');
        
        if (empty($guess)) {
            jsonError('Veuillez entrer un mot');
        }
        
        // Vérifier si la réponse est correcte (comparaison insensible à la casse)
        $civilianWord = $game['wordPair'][0];
        $isCorrect = (strtolower($guess) === strtolower($civilianWord));
        
        $game['mrWhiteGuess'] = [
            'guess' => $guess,
            'correct' => $isCorrect
        ];
        
        // Éliminer Mr. White dans tous les cas
        $game['players'][$playerId]['alive'] = false;
        
        if ($isCorrect) {
            // Mr. White gagne !
            $game['winner'] = 'mrwhite';
            $game['phase'] = 'ended';
        } else {
            // Vérifier si les civils gagnent maintenant
            $winner = checkVictory($game['players']);
            
            if ($winner) {
                $game['winner'] = $winner;
                $game['phase'] = 'ended';
            } else {
                $game['phase'] = 'reveal';
            }
        }
        
        $game['updatedAt'] = time();
        saveGame($code, $game);
        jsonResponse(['success' => true]);
        break;
        
    case 'next_round':
        if (!$isHost) {
            jsonError('Seul l\'hôte peut passer au tour suivant');
        }
        
        if ($game['phase'] !== 'reveal') {
            jsonError('Impossible de passer au tour suivant');
        }
        
        // Réinitialiser pour le nouveau tour
        foreach ($game['players'] as &$p) {
            $p['hasDescribed'] = false;
            $p['votes'] = 0;
            $p['votedFor'] = null;
        }
        unset($p);
        
        // Recréer l'ordre de description avec les joueurs vivants
        $alivePlayers = array_keys(array_filter($game['players'], fn($p) => $p['alive']));
        shuffle($alivePlayers);
        $game['descriptionOrder'] = $alivePlayers;
        
        $game['round']++;
        $game['phase'] = 'description';
        $game['currentDescriber'] = 0;
        $game['eliminatedPlayer'] = null;
        $game['mrWhiteGuess'] = null;
        $game['updatedAt'] = time();
        
        saveGame($code, $game);
        jsonResponse(['success' => true]);
        break;
        
    case 'restart':
        if (!$isHost) {
            jsonError('Seul l\'hôte peut relancer la partie');
        }
        
        // Réinitialiser tous les joueurs
        foreach ($game['players'] as &$p) {
            $p['role'] = null;
            $p['word'] = null;
            $p['alive'] = true;
            $p['hasDescribed'] = false;
            $p['votes'] = 0;
            $p['votedFor'] = null;
        }
        unset($p);
        
        // Réinitialiser la partie
        $game['status'] = 'waiting';
        $game['phase'] = 'lobby';
        $game['round'] = 0;
        $game['wordPair'] = null;
        $game['descriptionOrder'] = [];
        $game['currentDescriber'] = 0;
        $game['eliminatedPlayer'] = null;
        $game['winner'] = null;
        $game['mrWhiteGuess'] = null;
        $game['updatedAt'] = time();
        
        saveGame($code, $game);
        jsonResponse(['success' => true]);
        break;
    
    case 'change_words':
        if (!$isHost) {
            jsonError('Seul l\'hôte peut changer les mots');
        }
        
        if ($game['phase'] !== 'description' && $game['phase'] !== 'discussion') {
            jsonError('Impossible de changer les mots maintenant');
        }
        
        // Choisir une nouvelle paire de mots différente
        global $WORD_PAIRS;
        $currentPair = $game['wordPair'];
        $attempts = 0;
        
        do {
            $newPair = $WORD_PAIRS[array_rand($WORD_PAIRS)];
            if (random_int(0, 1) === 1) {
                $newPair = array_reverse($newPair);
            }
            $attempts++;
        } while ($newPair === $currentPair && $attempts < 10);
        
        $game['wordPair'] = $newPair;
        
        // Mettre à jour les mots des joueurs selon leur rôle
        foreach ($game['players'] as &$p) {
            if ($p['role'] === 'civilian') {
                $p['word'] = $newPair[0];
            } elseif ($p['role'] === 'undercover') {
                $p['word'] = $newPair[1];
            }
            // Mr. White garde word = null
            
            // Reset les descriptions pour recommencer
            $p['hasDescribed'] = false;
        }
        unset($p);
        
        // Remettre en phase description
        $game['phase'] = 'description';
        $game['currentDescriber'] = 0;
        
        // Remélanger l'ordre
        $alivePlayers = array_keys(array_filter($game['players'], fn($p) => $p['alive']));
        shuffle($alivePlayers);
        $game['descriptionOrder'] = $alivePlayers;
        
        $game['updatedAt'] = time();
        
        saveGame($code, $game);
        jsonResponse(['success' => true, 'message' => 'Nouveaux mots attribués !']);
        break;
        
    default:
        jsonError('Action inconnue');
}
